<?php

declare(strict_types=1);

use App\Admin\Auth;
use App\Admin\Security;
use App\Env;
use App\Http;

require_once __DIR__ . '/../../vendor/autoload.php';
\App\Env::bootstrap();

$nonce = Http::nonce();
Security::sendHeaders($nonce);

Auth::requireAuth();
Auth::requireAdmin();
$csrf = Auth::ensureCsrfToken();

function e(string $v): string
{
    return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
}

?><!doctype html>
<html lang="id">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="<?= e($csrf) ?>">
  <title>User Management | OG Shortlink Admin</title>
  <link rel="stylesheet" href="/admin/assets/admin.css" nonce="<?= e($nonce) ?>">
</head>
<body>
  <div class="app">
    <aside class="sidebar">
      <div class="brand">
        <div class="brand-left">
          <div class="logo">🔗</div>
          <div>
            <div class="brand-name">OG Shortlink</div>
            <div class="brand-sub">User Management</div>
          </div>
        </div>
      </div>

      <nav class="nav">
        <a href="/admin/index.php">Dashboard</a>
        <a href="/admin/shortlinks.php">Shortlinks</a>
        <a href="/admin/analytics.php">Analytics</a>
        <a href="/admin/users.php" aria-current="page">Users</a>
        <a href="/admin/domains.php">Domains</a>
        <a href="/admin/global-domains.php">Global Domains</a>
        <a href="/admin/cache-stats.php">Cache Stats</a>
        <a href="/admin/settings.php">Settings</a>
        <a href="/admin/logout.php">Logout</a>
      </nav>
    </aside>

    <main class="main">
      <div class="container">
        <div class="topbar">
          <div>
            <h1 class="page-title">User Management</h1>
            <div class="page-sub">Manage admin users with CRUD operations</div>
          </div>
        </div>

        <div class="grid">
          <!-- Form Card -->
          <div class="card panel">
            <div class="card-head">
              <h3 class="card-title" id="formTitle">Create User</h3>
            </div>

            <form id="frm" autocomplete="off">
              <div class="field">
                <label for="username">Username</label>
                <input type="text" id="username" name="username" placeholder="johndoe" required minlength="3">
              </div>

              <div class="field">
                <label for="password">Password</label>
                <input type="password" id="password" name="password" placeholder="Min 6 characters" required minlength="6">
              </div>

              <div style="display: flex; gap: 8px; margin-top: 10px;">
                <button class="btn btn-primary" id="submit" type="submit">Create User</button>
                <button class="btn" id="reset" type="button">Reset</button>
              </div>

              <div style="margin-top: 10px; font-size: 0.78rem;" class="muted">
                Password menggunakan bcrypt hashing untuk keamanan.
              </div>
            </form>
          </div>

          <!-- Table Card -->
          <div class="card panel">
            <div class="card-head">
              <div>
                <h3 class="card-title">Users List <span class="muted" id="meta"></span></h3>
                <div class="card-sub">All admin users</div>
              </div>
            </div>

            <!-- Bulk Actions Bar -->
            <div id="bulkActionsBar" style="display:none; gap:8px; margin-bottom:10px;">
              <button class="btn btn-danger" id="bulkDeleteBtn" type="button">
                Delete Selected (<span id="selectedCount">0</span>)
              </button>
              <button class="btn" id="clearSelectionBtn" type="button">Clear Selection</button>
            </div>

            <div class="table-wrap">
              <table id="tbl">
                <thead>
                  <tr>
                    <th style="width:40px;text-align:center;">
                      <input type="checkbox" id="selectAll" style="cursor:pointer;">
                    </th>
                    <th style="width:60px;">ID</th>
                    <th>Username</th>
                    <th style="width:100px;">Status</th>
                    <th style="width:120px;">Created</th>
                    <th style="width:200px;">Actions</th>
                  </tr>
                </thead>
                <tbody></tbody>
              </table>
            </div>
          </div>
        </div>

        <div class="footer">
          <p>&copy; <?= date('Y') ?> OG Shortlink</p>
        </div>
      </div>
    </main>
  </div>

  <script src="/admin/assets/toast.js" nonce="<?= e($nonce) ?>"></script>
  <script nonce="<?= e($nonce) ?>">
  (function(){
    'use strict';

    const csrf = document.querySelector('meta[name="csrf-token"]').getAttribute('content') || '';
    let editingId = 0;

    const elFormTitle = document.getElementById('formTitle');
    const elTblBody = document.querySelector('#tbl tbody');
    const elMeta = document.getElementById('meta');
    const frm = document.getElementById('frm');

    // Validate form element
    if (!frm || !(frm instanceof HTMLFormElement)) {
      console.error('Form element not found or invalid');
    }

    // Custom confirm dialog
    function customConfirm(message){
      return new Promise(function(resolve){
        var overlay = document.createElement('div');
        overlay.className = 'modal-overlay';
        overlay.setAttribute('data-open', '1');

        var modal = document.createElement('div');
        modal.className = 'modal';

        var modalBody = document.createElement('div');
        modalBody.className = 'modal-body';
        modalBody.textContent = message;

        var modalFooter = document.createElement('div');
        modalFooter.className = 'modal-footer';

        var btnCancel = document.createElement('button');
        btnCancel.textContent = 'Cancel';
        btnCancel.className = 'btn';
        btnCancel.onclick = function(){ overlay.remove(); resolve(false); };

        var btnConfirm = document.createElement('button');
        btnConfirm.textContent = 'Delete';
        btnConfirm.className = 'btn btn-danger';
        btnConfirm.onclick = function(){ overlay.remove(); resolve(true); };

        overlay.onclick = function(e){ if(e.target === overlay){ overlay.remove(); resolve(false); } };

        modalFooter.appendChild(btnCancel);
        modalFooter.appendChild(btnConfirm);
        modal.appendChild(modalBody);
        modal.appendChild(modalFooter);
        overlay.appendChild(modal);
        document.body.appendChild(overlay);

        btnConfirm.focus();
      });
    }

    function esc(s){
      return String(s)
        .replaceAll('&','&amp;')
        .replaceAll('<','&lt;')
        .replaceAll('>','&gt;')
        .replaceAll('"','&quot;')
        .replaceAll("'",'&#039;');
    }

    async function apiGet(url){
      const res = await fetch(url, { credentials: 'same-origin' });
      return await res.json();
    }

    async function apiPost(action, payload){
      const res = await fetch('/admin/api.php?action=' + encodeURIComponent(action), {
        method: 'POST',
        credentials: 'same-origin',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-Token': csrf
        },
        body: JSON.stringify(payload || {})
      });

      return await res.json();
    }

    function setFormModeCreate(){
      editingId = 0;
      elFormTitle.textContent = 'Create User';
      document.getElementById('username').disabled = false;
      document.getElementById('password').required = true;
      document.getElementById('password').placeholder = 'Min 6 characters';
      document.getElementById('submit').textContent = 'Create User';
      if (frm && typeof frm.reset === 'function') {
        frm.reset();
      } else {
        document.getElementById('username').value = '';
        document.getElementById('password').value = '';
      }
    }

    function fillForm(user){
      editingId = user.id;
      elFormTitle.textContent = 'Edit User: ' + user.username;
      document.getElementById('username').value = user.username;
      document.getElementById('username').disabled = true;
      document.getElementById('password').value = '';
      document.getElementById('password').required = false;
      document.getElementById('password').placeholder = 'Leave empty to keep current password';

      document.getElementById('submit').textContent = 'Update User';
    }

    function render(users){
      elTblBody.innerHTML = '';

      if (users.length === 0) {
        const tr = document.createElement('tr');
        tr.innerHTML = '<td colspan="6" style="text-align:center;"><span class="muted">No users found</span></td>';
        elTblBody.appendChild(tr);
        return;
      }

      for (const u of users){
        const statusBadge = u.is_active === 1
          ? '<span class="tag" style="background:#d1fae5;color:#065f46;">Active</span>'
          : '<span class="tag" style="background:#fee2e2;color:#991b1b;">Inactive</span>';

        const createdAt = new Date(u.created_at).toLocaleDateString('id-ID', {
          day: '2-digit',
          month: 'short',
          year: 'numeric'
        });

        const toggleBtnText = u.is_active === 1 ? 'Deactivate' : 'Activate';

        const tr = document.createElement('tr');
        tr.innerHTML = ''
          + '<td style="text-align:center;">'
          + '  <input type="checkbox" class="row-select" data-id="' + esc(u.id) + '" style="cursor:pointer;">'
          + '</td>'
          + '<td><strong>' + esc(u.id) + '</strong></td>'
          + '<td><strong>' + esc(u.username) + '</strong></td>'
          + '<td>' + statusBadge + '</td>'
          + '<td class="muted">' + createdAt + '</td>'
          + '<td>'
          + '  <div style="display:flex;gap:6px;">'
          + '    <button class="btn btn-sm" data-act="edit" data-id="' + esc(u.id) + '">Edit</button>'
          + '    <button class="btn btn-sm" data-act="toggle" data-id="' + esc(u.id) + '" data-active="' + u.is_active + '">' + toggleBtnText + '</button>'
          + '    <button class="btn btn-sm btn-danger" data-act="delete" data-id="' + esc(u.id) + '">Delete</button>'
          + '  </div>'
          + '</td>';

        elTblBody.appendChild(tr);
      }

      updateSelectionUI();
    }

    async function load(){
      try {
        const data = await apiGet('/admin/api.php?action=list_users');
        if (!data || data.ok !== true){
          elMeta.textContent = '(failed to load)';
          console.error('Failed to load users:', data);
          return;
        }

        const users = data.users || [];
        console.log('Loaded users:', users.length);
        render(users);
        elMeta.textContent = '(' + users.length + ')';
      } catch (e) {
        console.error('Load users error:', e);
        elMeta.textContent = '(error)';
      }
    }

    // Selection Management
    function getSelectedIds() {
      const checkboxes = document.querySelectorAll('.row-select:checked');
      return Array.from(checkboxes).map(cb => parseInt(cb.getAttribute('data-id'))).filter(id => !isNaN(id));
    }

    function updateSelectionUI() {
      const selectedIds = getSelectedIds();
      const count = selectedIds.length;

      // Update counter
      const selectedCountEl = document.getElementById('selectedCount');
      if (selectedCountEl) {
        selectedCountEl.textContent = String(count);
      }

      // Show/hide bulk actions bar
      const bulkActionsBar = document.getElementById('bulkActionsBar');
      if (bulkActionsBar) {
        if (count > 0) {
          bulkActionsBar.style.display = 'flex';
        } else {
          bulkActionsBar.style.display = 'none';
        }
      }

      // Update select all checkbox state
      const allCheckboxes = document.querySelectorAll('.row-select');
      const selectAllCheckbox = document.getElementById('selectAll');
      if (selectAllCheckbox) {
        if (allCheckboxes.length > 0) {
          selectAllCheckbox.checked = allCheckboxes.length === count;
          selectAllCheckbox.indeterminate = count > 0 && count < allCheckboxes.length;
        } else {
          selectAllCheckbox.checked = false;
          selectAllCheckbox.indeterminate = false;
        }
      }
    }

    function clearSelection() {
      document.querySelectorAll('.row-select:checked').forEach(cb => {
        cb.checked = false;
      });
      updateSelectionUI();
    }

    async function bulkDelete() {
      const selectedIds = getSelectedIds();
      if (selectedIds.length === 0) {
        showToast('No users selected', 'error');
        return;
      }

      const confirmed = await customConfirm(
        'Delete ' + selectedIds.length + ' user(s)?\n\n' +
        'IDs: ' + selectedIds.join(', ') + '\n\n' +
        'This action cannot be undone.'
      );

      if (!confirmed) return;

      const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
      if (!bulkDeleteBtn) return;

      bulkDeleteBtn.disabled = true;
      const originalHTML = bulkDeleteBtn.innerHTML;

      const selectedCountEl = document.getElementById('selectedCount');
      const currentCount = selectedCountEl ? selectedCountEl.textContent : '0';
      bulkDeleteBtn.innerHTML = 'Deleting <span id="selectedCount">' + currentCount + '</span> user(s)...';

      try {
        const result = await apiPost('bulk_delete_users', { ids: selectedIds });

        if (result && result.ok === true) {
          const deleted = result.deleted || 0;
          showToast('Deleted ' + deleted + ' user(s)', 'success');
          clearSelection();
          await load();
        } else {
          showToast(result.error || 'Bulk delete failed', 'error');
        }
      } catch (e) {
        showToast('Error during bulk delete: ' + e.message, 'error');
      } finally {
        bulkDeleteBtn.disabled = false;
        bulkDeleteBtn.innerHTML = originalHTML;
      }
    }

    // Event Listeners for Selection
    document.getElementById('selectAll').addEventListener('change', function() {
      const checkboxes = document.querySelectorAll('.row-select');
      checkboxes.forEach(cb => {
        cb.checked = this.checked;
      });
      updateSelectionUI();
    });

    document.getElementById('tbl').addEventListener('change', function(e) {
      if (e.target && e.target.classList.contains('row-select')) {
        updateSelectionUI();
      }
    });

    document.getElementById('clearSelectionBtn').addEventListener('click', function() {
      clearSelection();
    });

    document.getElementById('bulkDeleteBtn').addEventListener('click', function() {
      bulkDelete();
    });

    async function onTableClick(ev){
      const t = ev.target;
      if (!(t instanceof HTMLElement)) return;

      const act = t.getAttribute('data-act') || '';
      if (!act) return;

      const id = Number(t.getAttribute('data-id') || '0');
      if (id <= 0) return;

      if (act === 'edit'){
        try {
          const data = await apiGet('/admin/api.php?action=get_user&id=' + encodeURIComponent(String(id)));
          if (data && data.ok === true && data.user){
            fillForm(data.user);
            window.scrollTo({top: 0, behavior: 'smooth'});
            showToast('Editing user: ' + data.user.username, 'info');
          } else {
            showToast('Failed to load user', 'error');
          }
        } catch (e) {
          showToast('Error loading user', 'error');
        }
        return;
      }

      if (act === 'toggle'){
        const isActive = Number(t.getAttribute('data-active') || '0');
        const newStatus = isActive === 1 ? 0 : 1;

        try {
          t.disabled = true;
          const originalText = t.textContent;
          t.textContent = 'Updating...';

          const r = await apiPost('toggle_user_active', { id: id, is_active: newStatus });
          if (r && r.ok === true){
            showToast('User status updated', 'success');
            await load();
          } else {
            showToast(r.error || 'Failed to update user status', 'error');
            t.disabled = false;
            t.textContent = originalText;
          }
        } catch (e) {
          showToast('Error updating user status', 'error');
          t.disabled = false;
        }
        return;
      }

      if (act === 'delete'){
        const confirmed = await customConfirm('Delete this user?\n\nThis action cannot be undone.');
        if (!confirmed) return;

        try {
          t.disabled = true;
          t.textContent = 'Deleting...';

          const r = await apiPost('delete_user', { id: id });
          if (r && r.ok === true){
            showToast('User deleted successfully', 'success');
            await load();
          } else {
            showToast(r.error || 'Failed to delete user', 'error');
            t.disabled = false;
            t.textContent = 'Delete';
          }
        } catch (e) {
          showToast('Error deleting user', 'error');
          t.disabled = false;
          t.textContent = 'Delete';
        }
        return;
      }
    }

    frm.addEventListener('submit', async function(ev){
      ev.preventDefault();
      console.log('Form submitted, editingId:', editingId);

      const submitBtn = document.getElementById('submit');
      const username = (document.getElementById('username').value || '').trim();
      const password = (document.getElementById('password').value || '').trim();

      console.log('Username:', username, 'Password length:', password.length);

      // Client-side validation
      if (username.length < 3) {
        showToast('Username must be at least 3 characters', 'error');
        return;
      }

      if (!editingId && password.length < 6) {
        showToast('Password must be at least 6 characters', 'error');
        return;
      }

      const payload = {
        username: username,
        name: username,
        password: password
      };

      console.log('Payload:', payload);

      // Disable button during request
      submitBtn.disabled = true;
      const originalText = submitBtn.textContent;
      submitBtn.textContent = editingId ? 'Updating...' : 'Creating...';

      try {
        if (editingId){
          payload.id = editingId;
          console.log('Updating user ID:', editingId);
          const r = await apiPost('update_user', payload);
          console.log('Update response:', r);
          if (r && r.ok === true){
            showToast('User updated: ' + username, 'success');
            setFormModeCreate();
            await load();
            return;
          }
          showToast((r && r.error) ? r.error : 'Update failed', 'error');
          return;
        }

        console.log('Creating new user...');
        const r = await apiPost('create_user', payload);
        console.log('Create response:', r);
        if (r && r.ok === true){
          showToast('User created: ' + username, 'success');
          setFormModeCreate();
          await load();
          return;
        }

        showToast((r && r.error) ? r.error : 'Create failed', 'error');
      } catch (e) {
        console.error('Create user error:', e);
        showToast('Network error: ' + e.message, 'error');
      } finally {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
      }
    });

    document.getElementById('reset').addEventListener('click', function(){
      setFormModeCreate();
    });

    document.getElementById('tbl').addEventListener('click', onTableClick);

    console.log('Initializing users page...');
    console.log('Form element:', frm);
    console.log('CSRF token:', csrf);

    setFormModeCreate();
    load();
  })();
  </script>
</body>
</html>
