<?php

declare(strict_types=1);

use App\Db;
use App\Env;
use App\Http;
use App\ShortlinkRepository;

require_once __DIR__ . '/../../vendor/autoload.php';

header('Content-Type: application/json; charset=utf-8');

try {
    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
        http_response_code(405);
        echo json_encode(['ok' => false, 'error' => 'Method not allowed'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    // API Key Authentication
    $configuredKey = Env::getString('SHORTLINK_API_KEY', '');
    if ($configuredKey === '') {
        // API key not configured - fail closed (deny all requests)
        error_log('shortlink_api: SHORTLINK_API_KEY not configured, denying request');
        http_response_code(503);
        echo json_encode(['ok' => false, 'error' => 'API not configured'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $providedKey = (string) ($_SERVER['HTTP_X_SHORTLINK_KEY'] ?? '');
    if (!hash_equals($configuredKey, $providedKey)) {
        error_log('shortlink_api: Invalid API key attempt from ' . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
        http_response_code(401);
        echo json_encode(['ok' => false, 'error' => 'Unauthorized'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $raw = file_get_contents('php://input');
    $raw = is_string($raw) ? $raw : '';
    $payload = json_decode($raw, true);

    if (!is_array($payload)) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'Invalid JSON'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $pdo  = Db::pdo();
    $repo = new ShortlinkRepository($pdo);

    $code = (string) ($payload['code'] ?? '');
    $code = $code !== '' ? Http::validateCode($code) : generateCode($repo);

    if ($code === '') {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'Invalid code'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $target = Http::validateUrl((string) ($payload['target_url'] ?? ''), true);
    if ($target === '') {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'Invalid target_url'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $allowedHosts = Env::getList('SHORTLINK_ALLOWED_HOSTS');
    if (!Http::isAllowedHost($target, $allowedHosts)) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'Target host not allowed'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $ogTitle = trim((string) ($payload['og_title'] ?? ''));
    $ogDesc  = trim((string) ($payload['og_description'] ?? ''));
    if ($ogTitle === '' || $ogDesc === '') {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'og_title and og_description required'], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $ogType = trim((string) ($payload['og_type'] ?? 'website'));
    if ($ogType === '') {
        $ogType = 'website';
    }

    $ogSite = trim((string) ($payload['og_site_name'] ?? ''));
    $ogLoc  = trim((string) ($payload['og_locale'] ?? 'id_ID'));
    if ($ogLoc === '') {
        $ogLoc = 'id_ID';
    }

    $ogImgRaw = (string) ($payload['og_image_url'] ?? '');
    $ogImg    = $ogImgRaw !== '' ? Http::validateUrl($ogImgRaw, false) : '';

    $repo->insert([
        'code' => $code,
        'target_url' => $target,
        'og_title' => mb_substr($ogTitle, 0, 255),
        'og_description' => mb_substr($ogDesc, 0, 600),
        'og_image_url' => $ogImg,
        'og_type' => mb_substr($ogType, 0, 32),
        'og_site_name' => mb_substr($ogSite, 0, 64),
        'og_locale' => mb_substr($ogLoc, 0, 16),
    ]);

    $publicBase = rtrim(Env::getString('PUBLIC_BASE_URL', ''), '/');
    $shortUrl = $publicBase !== '' ? ($publicBase . '/' . $code) : $code;

    echo json_encode(['ok' => true, 'code' => $code, 'short_url' => $shortUrl], JSON_UNESCAPED_UNICODE);
    exit;
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => 'Server error'], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Keep it simple: generate sampai dapet yang belum kepakai.
 */
function generateCode(ShortlinkRepository $repo): string
{
    for ($i = 0; $i < 5; $i++) {
        $raw  = rtrim(strtr(base64_encode(random_bytes(6)), '+/', '-_'), '=');
        $code = substr($raw, 0, 10);

        $code = Http::validateCode($code);
        if ($code !== '' && !$repo->exists($code)) {
            return $code;
        }
    }

    return '';
}
