<?php
/**
 * Debug Script - DELETE AFTER USE
 * Upload to public/ and access via browser to diagnose 500 errors
 */

// Show all errors
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

echo "<pre style='font-family:monospace;background:#1a1a2e;color:#eee;padding:20px;'>";
echo "=== OG Shortlink Debug ===\n\n";

// 1. PHP Version
echo "1. PHP Version: " . PHP_VERSION . "\n";
echo "   Required: 8.1+\n";
echo "   Status: " . (version_compare(PHP_VERSION, '8.1.0', '>=') ? '✓ OK' : '✗ FAIL') . "\n\n";

// 2. Required Extensions
echo "2. Extensions:\n";
$required = ['pdo', 'pdo_mysql', 'curl', 'json', 'mbstring', 'openssl'];
foreach ($required as $ext) {
    $loaded = extension_loaded($ext);
    echo "   {$ext}: " . ($loaded ? '✓ OK' : '✗ MISSING') . "\n";
}
echo "\n";

// 3. Paths
$basePath = dirname(__DIR__);
echo "3. Paths:\n";
echo "   Base: {$basePath}\n";
echo "   Public: " . __DIR__ . "\n\n";

// 4. Critical Files
echo "4. Critical Files:\n";
$files = [
    '/.env' => 'Environment config',
    '/vendor/autoload.php' => 'Composer autoload',
    '/app/Env.php' => 'Env class',
    '/app/Db.php' => 'Database class',
    '/src/Infrastructure/Http/Request.php' => 'Request class',
    '/src/Presentation/Controller/RedirectController.php' => 'Main controller',
];

foreach ($files as $file => $desc) {
    $path = $basePath . $file;
    $exists = file_exists($path);
    echo "   {$file}: " . ($exists ? '✓ EXISTS' : '✗ MISSING') . " ({$desc})\n";
}
echo "\n";

// 5. Writable Directories
echo "5. Writable Directories:\n";
$dirs = ['/data', '/data/cache', '/data/logs'];
foreach ($dirs as $dir) {
    $path = $basePath . $dir;
    $exists = is_dir($path);
    $writable = $exists && is_writable($path);
    echo "   {$dir}: " . ($exists ? ($writable ? '✓ WRITABLE' : '✗ NOT WRITABLE') : '✗ MISSING') . "\n";
}
echo "\n";

// 6. .env Contents (sanitized)
echo "6. .env File:\n";
$envPath = $basePath . '/.env';
if (file_exists($envPath)) {
    echo "   ✓ EXISTS\n";
    $envContent = file_get_contents($envPath);
    // Check for key variables
    $checks = ['DB_DSN', 'DB_USER', 'PUBLIC_BASE_URL'];
    foreach ($checks as $key) {
        $found = strpos($envContent, $key) !== false;
        echo "   {$key}: " . ($found ? '✓ SET' : '✗ MISSING') . "\n";
    }
} else {
    echo "   ✗ MISSING - Run installer at /install/\n";
}
echo "\n";

// 7. Database Connection
echo "7. Database Connection:\n";
if (file_exists($basePath . '/.env') && file_exists($basePath . '/vendor/autoload.php')) {
    try {
        require_once $basePath . '/vendor/autoload.php';

        // Simple .env parser
        $envContent = file_get_contents($basePath . '/.env');
        $dsn = '';
        $user = '';
        $pass = '';

        foreach (explode("\n", $envContent) as $line) {
            $line = trim($line);
            if (strpos($line, 'DB_DSN=') === 0) {
                $dsn = trim(str_replace(['DB_DSN=', '"', "'"], '', $line));
            }
            if (strpos($line, 'DB_USER=') === 0) {
                $user = trim(str_replace(['DB_USER=', '"', "'"], '', $line));
            }
            if (strpos($line, 'DB_PASS=') === 0) {
                $pass = trim(str_replace(['DB_PASS=', '"', "'"], '', $line));
            }
        }

        if ($dsn) {
            $pdo = new PDO($dsn, $user, $pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_TIMEOUT => 5,
            ]);
            $pdo->query('SELECT 1');
            echo "   ✓ CONNECTED\n";

            // Check tables
            $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
            echo "   Tables: " . count($tables) . " found\n";
            if (in_array('shortlinks', $tables)) {
                echo "   shortlinks table: ✓ EXISTS\n";
            } else {
                echo "   shortlinks table: ✗ MISSING (run installer)\n";
            }
        } else {
            echo "   ✗ DB_DSN not found in .env\n";
        }
    } catch (PDOException $e) {
        echo "   ✗ CONNECTION FAILED\n";
        echo "   Error: " . $e->getMessage() . "\n";
    } catch (Throwable $e) {
        echo "   ✗ ERROR: " . $e->getMessage() . "\n";
    }
} else {
    echo "   ✗ Cannot test (missing .env or vendor)\n";
}
echo "\n";

// 8. Test Autoload
echo "8. Autoload Test:\n";
if (file_exists($basePath . '/vendor/autoload.php')) {
    try {
        require_once $basePath . '/vendor/autoload.php';

        $classes = [
            'App\\Env',
            'App\\Db',
            'App\\Cache',
            'App\\Infrastructure\\Http\\Request',
            'App\\Presentation\\Controller\\RedirectController',
        ];

        foreach ($classes as $class) {
            $exists = class_exists($class);
            $shortName = substr($class, strrpos($class, '\\') + 1);
            echo "   {$shortName}: " . ($exists ? '✓ OK' : '✗ NOT FOUND') . "\n";
        }
    } catch (Throwable $e) {
        echo "   ✗ Autoload error: " . $e->getMessage() . "\n";
    }
} else {
    echo "   ✗ vendor/autoload.php missing - run: composer install\n";
}
echo "\n";

// 9. Try to bootstrap
echo "9. Bootstrap Test:\n";
if (file_exists($basePath . '/vendor/autoload.php')) {
    try {
        if (!class_exists('App\\Env')) {
            require_once $basePath . '/vendor/autoload.php';
        }

        App\Env::bootstrap();
        echo "   ✓ Env::bootstrap() OK\n";

        // Test DB connection via Db class
        $pdo = App\Db::pdo();
        echo "   ✓ Db::pdo() OK\n";

    } catch (Throwable $e) {
        echo "   ✗ FAILED: " . $e->getMessage() . "\n";
        echo "   File: " . $e->getFile() . ":" . $e->getLine() . "\n";
    }
}
echo "\n";

// 10. Error Log
echo "10. Recent Errors:\n";
$errorLog = ini_get('error_log');
if ($errorLog && file_exists($errorLog) && is_readable($errorLog)) {
    $lines = file($errorLog);
    $recent = array_slice($lines, -10);
    foreach ($recent as $line) {
        echo "   " . trim($line) . "\n";
    }
} else {
    echo "   Error log not accessible\n";
    echo "   Check: {$basePath}/data/logs/ or server error log\n";
}

echo "\n=== END DEBUG ===\n";
echo "</pre>";

echo "<p style='color:red;font-weight:bold;'>⚠️ DELETE THIS FILE (debug.php) AFTER USE!</p>";
