<?php
/**
 * Fix Autoload - Regenerate composer autoload files
 * DELETE THIS FILE AFTER USE!
 */

ini_set('display_errors', '1');
error_reporting(E_ALL);

echo "<pre style='font-family:monospace;background:#1a1a2e;color:#eee;padding:20px;'>";
echo "=== Fix Autoload ===\n\n";

$basePath = dirname(__DIR__);
$vendorPath = $basePath . '/vendor';
$composerPath = $basePath . '/composer.json';

// Check composer.json
if (!file_exists($composerPath)) {
    die("ERROR: composer.json not found at {$composerPath}\n");
}

// Check vendor folder
if (!is_dir($vendorPath)) {
    die("ERROR: vendor folder not found. Run 'composer install' first.\n");
}

echo "Base path: {$basePath}\n";
echo "Vendor path: {$vendorPath}\n\n";

// Read composer.json
$composer = json_decode(file_get_contents($composerPath), true);
$autoload = $composer['autoload']['psr-4'] ?? [];

echo "PSR-4 Autoload config:\n";
print_r($autoload);
echo "\n";

// Check if src/ and app/ directories exist
$srcPath = $basePath . '/src';
$appPath = $basePath . '/app';

echo "Directory check:\n";
echo "  src/: " . (is_dir($srcPath) ? "EXISTS" : "MISSING") . "\n";
echo "  app/: " . (is_dir($appPath) ? "EXISTS" : "MISSING") . "\n\n";

// Check specific class files
$classFiles = [
    'App\\Infrastructure\\Http\\Request' => '/src/Infrastructure/Http/Request.php',
    'App\\Presentation\\Controller\\RedirectController' => '/src/Presentation/Controller/RedirectController.php',
    'App\\Env' => '/app/Env.php',
    'App\\Db' => '/app/Db.php',
];

echo "Class files check:\n";
foreach ($classFiles as $class => $file) {
    $fullPath = $basePath . $file;
    $exists = file_exists($fullPath);
    echo "  {$class}\n";
    echo "    File: {$file}\n";
    echo "    Status: " . ($exists ? "EXISTS" : "MISSING!") . "\n";
}
echo "\n";

// Try to regenerate autoload
echo "Regenerating autoload...\n";

// Method 1: Use composer if available
$composerPhar = $basePath . '/composer.phar';
$composerCmd = null;

if (file_exists($composerPhar)) {
    $composerCmd = "php {$composerPhar}";
} else {
    // Check if composer is in PATH
    exec('composer --version 2>&1', $output, $returnCode);
    if ($returnCode === 0) {
        $composerCmd = 'composer';
    }
}

if ($composerCmd) {
    echo "Using: {$composerCmd}\n";
    $cmd = "cd {$basePath} && {$composerCmd} dump-autoload -o 2>&1";
    exec($cmd, $output, $returnCode);
    echo "Output:\n" . implode("\n", $output) . "\n";
    echo "Return code: {$returnCode}\n";
} else {
    echo "Composer not found. Manual fix...\n\n";

    // Manual fix: regenerate autoload_classmap.php
    $classmap = [];

    // Scan app/ directory
    if (is_dir($appPath)) {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($appPath, RecursiveDirectoryIterator::SKIP_DOTS)
        );
        foreach ($iterator as $file) {
            if ($file->getExtension() === 'php') {
                $relativePath = str_replace($basePath . '/', '', $file->getPathname());
                $className = getClassFromFile($file->getPathname());
                if ($className) {
                    $classmap[$className] = '$baseDir . \'/' . str_replace('\\', '/', $relativePath) . '\'';
                    echo "  Found: {$className}\n";
                }
            }
        }
    }

    // Scan src/ directory
    if (is_dir($srcPath)) {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($srcPath, RecursiveDirectoryIterator::SKIP_DOTS)
        );
        foreach ($iterator as $file) {
            if ($file->getExtension() === 'php') {
                $relativePath = str_replace($basePath . '/', '', $file->getPathname());
                $className = getClassFromFile($file->getPathname());
                if ($className) {
                    $classmap[$className] = '$baseDir . \'/' . str_replace('\\', '/', $relativePath) . '\'';
                    echo "  Found: {$className}\n";
                }
            }
        }
    }

    echo "\nTotal classes found: " . count($classmap) . "\n";
}

echo "\n=== Testing Autoload ===\n";

// Clear any cached autoload
if (function_exists('opcache_reset')) {
    opcache_reset();
    echo "OPcache cleared.\n";
}

// Reload autoload
$autoloadFile = $vendorPath . '/autoload.php';
if (file_exists($autoloadFile)) {
    // Force reload
    $loader = require $autoloadFile;

    // Test classes
    $testClasses = [
        'App\\Env',
        'App\\Db',
        'App\\Cache',
        'App\\Infrastructure\\Http\\Request',
        'App\\Presentation\\Controller\\RedirectController',
    ];

    echo "\nClass loading test:\n";
    foreach ($testClasses as $class) {
        $exists = class_exists($class, true);
        $shortName = substr($class, strrpos($class, '\\') + 1);
        echo "  {$shortName}: " . ($exists ? "OK" : "FAILED") . "\n";
    }
}

echo "\n=== Done ===\n";
echo "</pre>";
echo "<p style='color:red;font-weight:bold;'>DELETE THIS FILE AFTER USE!</p>";

// Helper function to extract class name from PHP file
function getClassFromFile($file) {
    $content = file_get_contents($file);
    $namespace = '';
    $class = '';

    if (preg_match('/namespace\s+([^;]+);/', $content, $matches)) {
        $namespace = $matches[1];
    }

    if (preg_match('/class\s+(\w+)/', $content, $matches)) {
        $class = $matches[1];
    }

    if ($class) {
        return $namespace ? $namespace . '\\' . $class : $class;
    }

    return null;
}
