<?php

declare(strict_types=1);

require __DIR__ . '/../vendor/autoload.php';

use App\Env;
use App\Monitoring\HealthCheck;
use App\Database\ConnectionPool;

// Load .env file
Env::bootstrap();

header('Content-Type: application/json');

// Create health checker
$healthCheck = new HealthCheck();

// Database check
$healthCheck->addCheck('database', function () {
    $pdo = ConnectionPool::getInstance(
        Env::getString('DB_DSN', 'mysql:host=localhost;dbname=app'),
        Env::getString('DB_USER', 'root'),
        Env::getString('DB_PASS', '')
    )->getConnection();

    return HealthCheck::checkDatabase($pdo);
});

// Redis check
$healthCheck->addCheck('redis', function () {
    if (!extension_loaded('redis')) {
        return [
            'status' => 'skipped',
            'message' => 'Redis extension not installed',
        ];
    }

    $redis = new \Redis();
    $connected = $redis->connect(
        Env::getString('REDIS_HOST', '127.0.0.1'),
        Env::getInt('REDIS_PORT', 6379),
        2 // 2 second timeout
    );

    if (!$connected) {
        return [
            'status' => 'unhealthy',
            'error' => 'Failed to connect to Redis',
        ];
    }

    return HealthCheck::checkRedis($redis);
});

// Disk space check
$healthCheck->addCheck('disk', function () {
    return HealthCheck::checkDiskSpace('/', 80);
});

// Memory check
$healthCheck->addCheck('memory', function () {
    return HealthCheck::checkMemory(80);
});

// Run all checks
$result = $healthCheck->run();

// Set HTTP status based on health
http_response_code($result['status'] === 'healthy' ? 200 : 503);

// Output result
echo json_encode($result, JSON_PRETTY_PRINT | JSON_THROW_ON_ERROR);
