<?php

declare(strict_types=1);

/**
 * Shim Link (safe redirect + unwrap Meta/Facebook Family link shims)
 *
 * Supported Platforms:
 * - Facebook (l.facebook.com, lm.facebook.com, m.facebook.com)
 * - Instagram (l.instagram.com)
 * - WhatsApp (l.wl.co)
 * - Threads (l.threads.net)
 * - Messenger (l.messenger.com)
 *
 * Endpoint:
 *   /shim.php?d=<base64url>&sig=<hex_hmac_sha256>
 *
 * Where:
 *   d   = base64url_encode(destination_url)
 *   sig = hash_hmac('sha256', d, SHIM_HMAC_SECRET)
 *
 * Security:
 * - Signed param (anti tampering)
 * - Only http/https
 * - Blocks localhost/private/reserved IP as host
 * - Optional allowlist hosts/suffixes via ENV
 * - Recursive unwrapping (max depth: 3)
 */

header('Content-Type: text/plain; charset=utf-8');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('Referrer-Policy: strict-origin-when-cross-origin');
header('Cache-Control: no-store, max-age=0');

$secret = (string) getenv('SHIM_HMAC_SECRET');
if ($secret === '') {
    http_response_code(500);
    echo "Server misconfigured.\n";
    exit;
}

$d = (string) ($_GET['d'] ?? '');
$sig = (string) ($_GET['sig'] ?? '');

if ($d === '' || $sig === '') {
    http_response_code(400);
    echo "Bad request.\n";
    exit;
}

$expected = hash_hmac('sha256', $d, $secret);
if (!hash_equals($expected, $sig)) {
    http_response_code(400);
    echo "Invalid signature.\n";
    exit;
}

$rawUrl = base64urlDecode($d);
if ($rawUrl === null) {
    http_response_code(400);
    echo "Invalid payload.\n";
    exit;
}

$finalUrl = unwrapKnownShims($rawUrl, 0);

$validated = validateDestinationUrl($finalUrl);
if ($validated['ok'] !== true) {
    http_response_code(400);
    echo "Blocked.\n";
    exit;
}

header('Location: ' . $finalUrl, true, 302);
exit;

/**
 * @return array{ok: bool, reason?: string}
 */
function validateDestinationUrl(string $url): array
{
    $parts = parse_url($url);
    if (!is_array($parts)) {
        return ['ok' => false, 'reason' => 'parse_failed'];
    }

    $scheme = strtolower((string) ($parts['scheme'] ?? ''));
    if ($scheme !== 'http' && $scheme !== 'https') {
        return ['ok' => false, 'reason' => 'bad_scheme'];
    }

    $host = strtolower((string) ($parts['host'] ?? ''));
    if ($host === '') {
        return ['ok' => false, 'reason' => 'no_host'];
    }

    if (isset($parts['user']) || isset($parts['pass'])) {
        return ['ok' => false, 'reason' => 'userinfo_forbidden'];
    }

    if ($host === 'localhost' || $host === 'localhost.localdomain') {
        return ['ok' => false, 'reason' => 'localhost_blocked'];
    }

    if (filter_var($host, FILTER_VALIDATE_IP) !== false) {
        if (!isPublicIp($host)) {
            return ['ok' => false, 'reason' => 'ip_not_public'];
        }
    }

    if (!isAllowedHost($host)) {
        return ['ok' => false, 'reason' => 'host_not_allowed'];
    }

    return ['ok' => true];
}

function isPublicIp(string $ip): bool
{
    $ok = filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
    return $ok !== false;
}

function isAllowedHost(string $host): bool
{
    $allowHosts = trim((string) getenv('SHIM_ALLOWED_HOSTS'));
    $allowSuf   = trim((string) getenv('SHIM_ALLOWED_SUFFIXES'));

    if ($allowHosts === '' && $allowSuf === '') {
        // Default: allow all (not recommended, but deterministic)
        return true;
    }

    if ($allowHosts !== '') {
        $list = explode(',', $allowHosts);
        foreach ($list as $item) {
            $item = strtolower(trim($item));
            if ($item !== '' && $host === $item) {
                return true;
            }
        }
    }

    if ($allowSuf !== '') {
        $list = explode(',', $allowSuf);
        foreach ($list as $suf) {
            $suf = strtolower(trim($suf));
            if ($suf === '') {
                continue;
            }

            // suffix like ".example.com" allows foo.example.com, not example.com unless explicitly included
            if (str_starts_with($suf, '.') && str_ends_with($host, $suf)) {
                return true;
            }

            // exact fallback
            if ($host === $suf) {
                return true;
            }
        }
    }

    return false;
}

/**
 * Unwrap known shims (FB/IG) to the actual destination.
 * Depth-limited to avoid loops.
 */
function unwrapKnownShims(string $url, int $depth): string
{
    if ($depth >= 3) {
        return $url;
    }

    $parts = parse_url($url);
    if (!is_array($parts)) {
        return $url;
    }

    $host = strtolower((string) ($parts['host'] ?? ''));
    $path = (string) ($parts['path'] ?? '');

    if ($host === '') {
        return $url;
    }

    parse_str((string) ($parts['query'] ?? ''), $q);

    // Facebook link shim commonly uses l.facebook.com/l.php?u=<encoded>
    if (isFacebookShimHost($host) && (str_ends_with($path, '/l.php') || str_ends_with($path, '/l/'))) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }

        return $url;
    }

    // Instagram link shim commonly uses l.instagram.com/?u=<encoded>
    if ($host === 'l.instagram.com') {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }

        return $url;
    }

    // WhatsApp link shim commonly uses l.wl.co/l?u=<encoded>
    if ($host === 'l.wl.co' && str_ends_with($path, '/l')) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }

        return $url;
    }

    // Threads link shim commonly uses l.threads.net/?u=<encoded>
    if ($host === 'l.threads.net' || str_ends_with($host, '.threads.net')) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }

        return $url;
    }

    // Messenger link shim (part of Facebook family)
    if ($host === 'l.messenger.com' || str_ends_with($host, '.messenger.com')) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }

        return $url;
    }

    return $url;
}

function isFacebookShimHost(string $host): bool
{
    return $host === 'l.facebook.com' || $host === 'lm.facebook.com' || $host === 'm.facebook.com';
}

/**
 * @return string|null
 */
function decodeShimParamUrl(string $value): ?string
{
    if ($value === '') {
        return null;
    }

    $decoded = rawurldecode($value);
    $decoded = trim($decoded);

    if ($decoded === '') {
        return null;
    }

    // Basic sanity: must look like a URL
    $p = parse_url($decoded);
    if (!is_array($p)) {
        return null;
    }

    $scheme = strtolower((string) ($p['scheme'] ?? ''));
    if ($scheme !== 'http' && $scheme !== 'https') {
        return null;
    }

    return $decoded;
}

/**
 * @return string|null
 */
function base64urlDecode(string $data): ?string
{
    $data = str_replace(['-', '_'], ['+', '/'], $data);
    $pad = strlen($data) % 4;
    if ($pad !== 0) {
        $data .= str_repeat('=', 4 - $pad);
    }

    $out = base64_decode($data, true);
    if ($out === false) {
        return null;
    }

    return $out;
}
