<?php

declare(strict_types=1);

/**
 * Professional Shim Link (Long-lasting, Production-ready)
 *
 * Features:
 * ✅ Decision layer untuk routing logic
 * ✅ Bot & human separation
 * ✅ Logging minimal (privacy-aware)
 * ✅ Fallback statis (error recovery)
 * ✅ Domain reusable (multi-purpose)
 *
 * Supported Platforms:
 * - Facebook (l.facebook.com, lm.facebook.com, m.facebook.com)
 * - Instagram (l.instagram.com)
 * - WhatsApp (l.wl.co)
 * - Threads (l.threads.net)
 * - Messenger (l.messenger.com)
 *
 * Endpoint:
 *   /shim_v2.php?d=<base64url>&sig=<base64url_hmac_sha256>
 *
 * Security:
 * - HMAC-SHA256 signature verification (Base64url encoded, URL-safe)
 * - Bot/human traffic separation
 * - Privacy-aware logging (no PII)
 * - Static fallback page
 * - Rate limiting per IP
 */

use App\Http;
use App\Cache;

require_once __DIR__ . '/../vendor/autoload.php';

// ============================================================
// 1. SECURITY HEADERS
// ============================================================

function sendSecurityHeaders(bool $isBot): void
{
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: DENY');
    header('Referrer-Policy: no-referrer');

    if ($isBot) {
        // For bots: Allow caching (reduce load)
        header('Cache-Control: public, max-age=300, must-revalidate');
        header('Vary: User-Agent');
    } else {
        // For humans: No cache (privacy)
        header('Cache-Control: no-store, no-cache, must-revalidate, private');
    }
}

// ============================================================
// 2. DECISION LAYER
// ============================================================

/**
 * Decision layer untuk routing logic
 *
 * @param string $url Destination URL
 * @param bool $isBot Is request from bot
 * @return array{action: string, url: string, reason: string}
 */
function makeRoutingDecision(string $url, bool $isBot): array
{
    // Decision 1: Validate destination URL
    $validation = validateDestinationUrl($url);
    if ($validation['ok'] !== true) {
        return [
            'action' => 'show_fallback',
            'url' => '',
            'reason' => $validation['reason'] ?? 'validation_failed',
        ];
    }

    // Decision 2: Bot vs Human routing
    if ($isBot) {
        // For bots: Direct redirect (no intermediate page)
        return [
            'action' => 'direct_redirect',
            'url' => $url,
            'reason' => 'bot_traffic',
        ];
    }

    // Decision 3: Human traffic - unwrap shims first
    $unwrapped = unwrapKnownShims($url, 0);

    // Decision 4: Final validation after unwrap
    $finalValidation = validateDestinationUrl($unwrapped);
    if ($finalValidation['ok'] !== true) {
        return [
            'action' => 'show_fallback',
            'url' => '',
            'reason' => $finalValidation['reason'] ?? 'unwrap_validation_failed',
        ];
    }

    return [
        'action' => 'direct_redirect',
        'url' => $unwrapped,
        'reason' => 'human_traffic',
    ];
}

// ============================================================
// 3. PRIVACY-AWARE MINIMAL LOGGING
// ============================================================

/**
 * Log shim events (privacy-aware, no PII)
 *
 * @param string $event Event type
 * @param array<string, mixed> $context Context data (sanitized)
 */
function logShimEvent(string $event, array $context = []): void
{
    // Privacy: Redact sensitive data
    $sanitized = [];
    foreach ($context as $key => $value) {
        // Skip PII
        if (in_array($key, ['ip', 'user_agent', 'url'], true)) {
            continue;
        }
        $sanitized[$key] = $value;
    }

    $log = [
        'timestamp' => date('c'),
        'event' => $event,
        'context' => $sanitized,
    ];

    // Log to error_log (minimal, structured)
    error_log('shim_event: ' . json_encode($log));

    // Optional: Store in Cache for metrics (aggregated only)
    $metricKey = "shim_metric:{$event}";
    Cache::increment($metricKey, 3600); // 1 hour TTL
}

// ============================================================
// 4. STATIC FALLBACK PAGE
// ============================================================

/**
 * Show static fallback page (error recovery)
 *
 * @param string $reason Error reason (internal only)
 */
function showFallbackPage(string $reason): void
{
    http_response_code(400);
    header('Content-Type: text/html; charset=utf-8');

    $nonce = base64_encode(random_bytes(16));
    header("Content-Security-Policy: default-src 'none'; style-src 'nonce-{$nonce}'; frame-ancestors 'none'");

    echo '<!DOCTYPE html>';
    echo '<html lang="id"><head>';
    echo '<meta charset="utf-8">';
    echo '<meta name="viewport" content="width=device-width, initial-scale=1">';
    echo '<title>Link Not Available</title>';
    echo '<style nonce="' . htmlspecialchars($nonce, ENT_QUOTES, 'UTF-8') . '">';
    echo 'body{font-family:system-ui,-apple-system,sans-serif;margin:0;padding:24px;background:#f9fafb;color:#111}';
    echo '.container{max-width:480px;margin:0 auto;padding:32px;background:#fff;border-radius:12px;box-shadow:0 1px 3px rgba(0,0,0,.1)}';
    echo 'h1{margin:0 0 16px;font-size:24px;color:#dc2626}';
    echo 'p{margin:0 0 12px;line-height:1.6;color:#4b5563}';
    echo '.code{font-family:monospace;font-size:12px;color:#6b7280;margin-top:24px;padding-top:16px;border-top:1px solid #e5e7eb}';
    echo '</style>';
    echo '</head><body>';
    echo '<div class="container">';
    echo '<h1>Link Not Available</h1>';
    echo '<p>The link you followed is not accessible at this time.</p>';
    echo '<p>This could be due to:</p>';
    echo '<ul style="margin:8px 0;padding-left:24px;color:#6b7280">';
    echo '<li>Invalid or expired link</li>';
    echo '<li>Security restrictions</li>';
    echo '<li>Destination unavailable</li>';
    echo '</ul>';
    echo '<p style="margin-top:24px;font-size:14px">If you believe this is an error, please contact support.</p>';

    // Only log reason internally, don't expose to user
    logShimEvent('fallback_shown', ['reason' => $reason]);

    echo '</div>';
    echo '</body></html>';
    exit;
}

// ============================================================
// 5. RATE LIMITING (DOMAIN REUSABLE)
// ============================================================

/**
 * Check rate limit per IP (prevent abuse)
 *
 * @param string $ip Client IP
 * @return bool True if within limit, false if exceeded
 */
function checkRateLimit(string $ip): bool
{
    $key = "shim_rate:" . hash('sha256', $ip); // Hash IP for privacy
    $requests = Cache::fetch($key, $success) ?: [];
    $now = time();

    // Clean old requests (60 second window)
    $requests = array_filter($requests, fn($ts) => $ts > ($now - 60));

    // Max: 60 requests per minute per IP
    if (count($requests) >= 60) {
        logShimEvent('rate_limit_exceeded', ['timestamp' => $now]);
        return false;
    }

    $requests[] = $now;
    Cache::store($key, $requests, 60);

    return true;
}

// ============================================================
// 6. MAIN EXECUTION
// ============================================================

try {
    // Step 1: Get User-Agent for bot detection
    $ua = (string) ($_SERVER['HTTP_USER_AGENT'] ?? '');
    $isBot = Http::isFacebookBot($ua);

    // Step 2: Send appropriate headers
    sendSecurityHeaders($isBot);

    // Step 3: Get client IP
    $clientIp = (string) ($_SERVER['REMOTE_ADDR'] ?? '0.0.0.0');

    // Step 4: Rate limiting (domain reusable - prevent abuse)
    if (!checkRateLimit($clientIp)) {
        http_response_code(429);
        header('Retry-After: 60');
        echo "Too many requests. Please try again later.\n";
        exit;
    }

    // Step 5: Verify HMAC signature
    $secret = (string) getenv('SHIM_HMAC_SECRET');
    if ($secret === '') {
        logShimEvent('secret_not_configured');
        showFallbackPage('secret_missing');
    }

    $d = (string) ($_GET['d'] ?? '');
    $sig = (string) ($_GET['sig'] ?? '');

    if ($d === '' || $sig === '') {
        logShimEvent('invalid_request', ['has_d' => $d !== '', 'has_sig' => $sig !== '']);
        showFallbackPage('invalid_params');
    }

    // Generate expected signature (binary HMAC -> base64url)
    $expectedRaw = hash_hmac('sha256', $d, $secret, true);
    $expectedB64 = base64urlEncode($expectedRaw);

    if (!hash_equals($expectedB64, $sig)) {
        logShimEvent('signature_mismatch');
        showFallbackPage('invalid_signature');
    }

    // Step 6: Decode payload
    $rawUrl = base64urlDecode($d);
    if ($rawUrl === null) {
        logShimEvent('decode_failed');
        showFallbackPage('decode_error');
    }

    // Step 7: Decision layer - routing logic
    $decision = makeRoutingDecision($rawUrl, $isBot);

    logShimEvent('routing_decision', [
        'action' => $decision['action'],
        'reason' => $decision['reason'],
        'is_bot' => $isBot,
    ]);

    // Step 8: Execute decision
    if ($decision['action'] === 'show_fallback') {
        showFallbackPage($decision['reason']);
    }

    if ($decision['action'] === 'direct_redirect') {
        header('Location: ' . $decision['url'], true, 302);
        exit;
    }

    // Fallback (should not reach here)
    showFallbackPage('unknown_action');

} catch (Throwable $e) {
    $rid = bin2hex(random_bytes(8));
    error_log('shim_error rid=' . $rid . ' type=' . get_class($e) . ' msg=' . $e->getMessage());

    logShimEvent('exception', ['request_id' => $rid]);
    showFallbackPage('internal_error');
}

// ============================================================
// 7. HELPER FUNCTIONS (same as original shim.php)
// ============================================================

function base64urlEncode(string $data): string
{
    $b64 = base64_encode($data);
    $b64url = strtr($b64, '+/', '-_');
    return rtrim($b64url, '=');
}

function base64urlDecode(string $s): ?string
{
    // Convert from URL-safe variant
    $s = strtr($s, '-_', '+/');

    // Add padding if needed (RFC 4648)
    $remainder = strlen($s) % 4;
    if ($remainder > 0) {
        $s .= str_repeat('=', 4 - $remainder);
    }

    $decoded = base64_decode($s, true);
    return ($decoded !== false) ? $decoded : null;
}

/**
 * @return array{ok: bool, reason?: string}
 */
function validateDestinationUrl(string $url): array
{
    $parts = parse_url($url);
    if (!is_array($parts)) {
        return ['ok' => false, 'reason' => 'parse_failed'];
    }

    $scheme = strtolower((string) ($parts['scheme'] ?? ''));
    if ($scheme !== 'http' && $scheme !== 'https') {
        return ['ok' => false, 'reason' => 'bad_scheme'];
    }

    $host = strtolower((string) ($parts['host'] ?? ''));
    if ($host === '') {
        return ['ok' => false, 'reason' => 'no_host'];
    }

    if (isset($parts['user']) || isset($parts['pass'])) {
        return ['ok' => false, 'reason' => 'userinfo_forbidden'];
    }

    if ($host === 'localhost' || $host === 'localhost.localdomain') {
        return ['ok' => false, 'reason' => 'localhost_blocked'];
    }

    if (filter_var($host, FILTER_VALIDATE_IP) !== false) {
        if (!isPublicIp($host)) {
            return ['ok' => false, 'reason' => 'ip_not_public'];
        }
    }

    if (!isAllowedHost($host)) {
        return ['ok' => false, 'reason' => 'host_not_allowed'];
    }

    return ['ok' => true];
}

function isPublicIp(string $ip): bool
{
    $ok = filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
    return $ok !== false;
}

function isAllowedHost(string $host): bool
{
    $allowHosts = trim((string) getenv('SHIM_ALLOWED_HOSTS'));
    $allowSuf   = trim((string) getenv('SHIM_ALLOWED_SUFFIXES'));

    if ($allowHosts === '' && $allowSuf === '') {
        return true;
    }

    if ($allowHosts !== '') {
        $list = explode(',', $allowHosts);
        foreach ($list as $item) {
            $item = strtolower(trim($item));
            if ($item !== '' && $host === $item) {
                return true;
            }
        }
    }

    if ($allowSuf !== '') {
        $list = explode(',', $allowSuf);
        foreach ($list as $suf) {
            $suf = strtolower(trim($suf));
            if ($suf === '') {
                continue;
            }

            if (str_starts_with($suf, '.') && str_ends_with($host, $suf)) {
                return true;
            }

            if ($host === $suf) {
                return true;
            }
        }
    }

    return false;
}

function unwrapKnownShims(string $url, int $depth): string
{
    if ($depth >= 3) {
        return $url;
    }

    $parts = parse_url($url);
    if (!is_array($parts)) {
        return $url;
    }

    $host = strtolower((string) ($parts['host'] ?? ''));
    $path = (string) ($parts['path'] ?? '');

    if ($host === '') {
        return $url;
    }

    parse_str((string) ($parts['query'] ?? ''), $q);

    // Facebook
    if (isFacebookShimHost($host) && (str_ends_with($path, '/l.php') || str_ends_with($path, '/l/'))) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }
        return $url;
    }

    // Instagram
    if ($host === 'l.instagram.com') {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }
        return $url;
    }

    // WhatsApp
    if ($host === 'l.wl.co' && str_ends_with($path, '/l')) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }
        return $url;
    }

    // Threads
    if ($host === 'l.threads.net' || str_ends_with($host, '.threads.net')) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }
        return $url;
    }

    // Messenger
    if ($host === 'l.messenger.com' || str_ends_with($host, '.messenger.com')) {
        $u = (string) ($q['u'] ?? '');
        $decoded = decodeShimParamUrl($u);
        if ($decoded !== null) {
            return unwrapKnownShims($decoded, $depth + 1);
        }
        return $url;
    }

    return $url;
}

function isFacebookShimHost(string $host): bool
{
    $fbHosts = [
        'l.facebook.com',
        'lm.facebook.com',
        'm.facebook.com',
    ];

    return in_array($host, $fbHosts, true);
}

function decodeShimParamUrl(string $param): ?string
{
    if ($param === '') {
        return null;
    }

    $decoded = urldecode($param);

    if (filter_var($decoded, FILTER_VALIDATE_URL) !== false) {
        return $decoded;
    }

    return null;
}
