<?php

declare(strict_types=1);

/**
 * Session & CSRF Token Test Script
 *
 * Test URL: http://localhost:8080/test-session.php
 */

require_once __DIR__ . '/../vendor/autoload.php';
\App\Env::bootstrap();

use App\Admin\Auth;

header('Content-Type: text/html; charset=utf-8');

?><!DOCTYPE html>
<html>
<head>
    <title>Session Test</title>
    <style>
        body { font-family: monospace; padding: 20px; background: #f5f5f5; }
        .box { background: white; padding: 20px; margin: 10px 0; border-radius: 8px; border-left: 4px solid #3b82f6; }
        .success { border-left-color: #10b981; }
        .error { border-left-color: #ef4444; }
        .warning { border-left-color: #f59e0b; }
        pre { background: #1e293b; color: #e2e8f0; padding: 15px; border-radius: 4px; overflow-x: auto; }
        h2 { margin-top: 0; color: #1e293b; }
        table { width: 100%; border-collapse: collapse; }
        td { padding: 8px; border-bottom: 1px solid #e5e7eb; }
        td:first-child { font-weight: bold; width: 200px; color: #64748b; }
        .btn { display: inline-block; padding: 10px 20px; background: #3b82f6; color: white; text-decoration: none; border-radius: 6px; margin: 5px 0; }
        .btn:hover { background: #2563eb; }
    </style>
</head>
<body>
    <h1>🔐 Session & CSRF Token Test</h1>

    <?php
    // Initialize session
    Auth::initSession();
    $csrf = Auth::ensureCsrfToken();

    // Session info
    $sessionId = session_id();
    $sessionName = session_name();
    $sessionStatus = session_status();
    $sessionPath = session_save_path();

    // Cookie info
    $cookieParams = session_get_cookie_params();
    $hasCookie = isset($_COOKIE[$sessionName]);

    // Environment
    $sessionDriver = \App\Env::getString('SESSION_DRIVER', 'file');
    $appDebug = \App\Env::getString('APP_DEBUG', 'false');
    ?>

    <div class="box <?= $sessionStatus === PHP_SESSION_ACTIVE ? 'success' : 'error' ?>">
        <h2>Session Status</h2>
        <table>
            <tr>
                <td>Status:</td>
                <td>
                    <?php if ($sessionStatus === PHP_SESSION_ACTIVE): ?>
                        <strong style="color: #10b981;">✓ ACTIVE</strong>
                    <?php else: ?>
                        <strong style="color: #ef4444;">✗ INACTIVE</strong>
                    <?php endif; ?>
                </td>
            </tr>
            <tr>
                <td>Session ID:</td>
                <td><code><?= htmlspecialchars($sessionId) ?></code></td>
            </tr>
            <tr>
                <td>Session Name:</td>
                <td><code><?= htmlspecialchars($sessionName) ?></code></td>
            </tr>
            <tr>
                <td>Session Driver:</td>
                <td><code><?= htmlspecialchars($sessionDriver) ?></code></td>
            </tr>
            <tr>
                <td>Save Path:</td>
                <td><code><?= htmlspecialchars($sessionPath) ?></code></td>
            </tr>
            <tr>
                <td>Save Path Exists:</td>
                <td>
                    <?php if (is_dir($sessionPath)): ?>
                        <strong style="color: #10b981;">✓ YES</strong>
                    <?php else: ?>
                        <strong style="color: #ef4444;">✗ NO</strong>
                    <?php endif; ?>
                </td>
            </tr>
            <tr>
                <td>Save Path Writable:</td>
                <td>
                    <?php if (is_writable($sessionPath)): ?>
                        <strong style="color: #10b981;">✓ YES</strong>
                    <?php else: ?>
                        <strong style="color: #ef4444;">✗ NO</strong>
                    <?php endif; ?>
                </td>
            </tr>
        </table>
    </div>

    <div class="box <?= $hasCookie ? 'success' : 'warning' ?>">
        <h2>Cookie Configuration</h2>
        <table>
            <tr>
                <td>Cookie Sent:</td>
                <td>
                    <?php if ($hasCookie): ?>
                        <strong style="color: #10b981;">✓ YES</strong>
                    <?php else: ?>
                        <strong style="color: #f59e0b;">⚠ NO (First request)</strong>
                    <?php endif; ?>
                </td>
            </tr>
            <tr>
                <td>Cookie Value:</td>
                <td><code><?= htmlspecialchars($_COOKIE[$sessionName] ?? 'N/A') ?></code></td>
            </tr>
            <tr>
                <td>Path:</td>
                <td><code><?= htmlspecialchars($cookieParams['path']) ?></code></td>
            </tr>
            <tr>
                <td>Secure:</td>
                <td><?= $cookieParams['secure'] ? '✓ Yes' : '✗ No' ?></td>
            </tr>
            <tr>
                <td>HttpOnly:</td>
                <td><?= $cookieParams['httponly'] ? '✓ Yes' : '✗ No' ?></td>
            </tr>
            <tr>
                <td>SameSite:</td>
                <td><code><?= htmlspecialchars($cookieParams['samesite']) ?></code></td>
            </tr>
        </table>
    </div>

    <div class="box success">
        <h2>CSRF Token</h2>
        <table>
            <tr>
                <td>Token:</td>
                <td><code><?= htmlspecialchars($csrf) ?></code></td>
            </tr>
            <tr>
                <td>Length:</td>
                <td><?= strlen($csrf) ?> characters</td>
            </tr>
            <tr>
                <td>In Session:</td>
                <td>
                    <?php if (isset($_SESSION['csrf_token'])): ?>
                        <strong style="color: #10b981;">✓ YES</strong>
                    <?php else: ?>
                        <strong style="color: #ef4444;">✗ NO</strong>
                    <?php endif; ?>
                </td>
            </tr>
        </table>
    </div>

    <div class="box">
        <h2>Session Data</h2>
        <pre><?= htmlspecialchars(json_encode($_SESSION, JSON_PRETTY_PRINT)) ?></pre>
    </div>

    <div class="box">
        <h2>Environment</h2>
        <table>
            <tr>
                <td>APP_DEBUG:</td>
                <td><code><?= htmlspecialchars($appDebug) ?></code></td>
            </tr>
            <tr>
                <td>APP_ENV:</td>
                <td><code><?= htmlspecialchars(\App\Env::getString('APP_ENV', 'production')) ?></code></td>
            </tr>
            <tr>
                <td>HTTPS:</td>
                <td><?= !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? '✓ Yes' : '✗ No' ?></td>
            </tr>
        </table>
    </div>

    <div class="box">
        <h2>Session Files</h2>
        <?php
        $files = glob($sessionPath . '/sess_*');
        if ($files && count($files) > 0):
        ?>
            <p>Found <?= count($files) ?> session file(s):</p>
            <pre><?php
            foreach (array_slice($files, 0, 5) as $file) {
                echo basename($file) . ' (' . filesize($file) . ' bytes)' . "\n";
            }
            if (count($files) > 5) {
                echo '... and ' . (count($files) - 5) . ' more';
            }
            ?></pre>
        <?php else: ?>
            <p style="color: #f59e0b;">⚠ No session files found</p>
        <?php endif; ?>
    </div>

    <div class="box">
        <h2>Actions</h2>
        <a href="/test-session.php" class="btn">🔄 Refresh</a>
        <a href="/admin/login.php" class="btn">🔐 Go to Login</a>
    </div>

    <p style="color: #64748b; font-size: 12px;">
        Generated at <?= date('Y-m-d H:i:s') ?> •
        Session ID: <?= substr($sessionId, 0, 8) ?>...
    </p>
</body>
</html>
