<?php
/**
 * OG Shortlink - Quick Setup
 * Single-file installer for when /install/ folder is missing
 *
 * Upload to: public_html/setup.php
 * Access: https://yourdomain.com/setup.php
 * DELETE AFTER USE!
 */

declare(strict_types=1);

ini_set('display_errors', '1');
error_reporting(E_ALL);

session_start();

$basePath = __DIR__;
$step = (int) ($_GET['step'] ?? $_POST['step'] ?? 1);
$error = null;
$success = null;

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'test_db') {
        header('Content-Type: application/json');
        try {
            $dsn = "mysql:host={$_POST['db_host']};port={$_POST['db_port']};dbname={$_POST['db_name']};charset=utf8mb4";
            $pdo = new PDO($dsn, $_POST['db_user'], $_POST['db_pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            ]);
            $pdo->query('SELECT 1');
            echo json_encode(['success' => true, 'message' => 'Connection successful!']);
        } catch (PDOException $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }

    if ($action === 'install') {
        try {
            // 1. Test database connection
            $dsn = "mysql:host={$_POST['db_host']};port={$_POST['db_port']};dbname={$_POST['db_name']};charset=utf8mb4";
            $pdo = new PDO($dsn, $_POST['db_user'], $_POST['db_pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            ]);

            // 2. Install schema - create tables manually (more reliable)
            $tables = [
                // Users table
                "CREATE TABLE IF NOT EXISTS `users` (
                    `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                    `username` varchar(100) NOT NULL,
                    `password_hash` varchar(255) NOT NULL,
                    `name` varchar(255) NOT NULL,
                    `role` enum('admin','user') NOT NULL DEFAULT 'user',
                    `is_active` tinyint(1) NOT NULL DEFAULT 1,
                    `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
                    `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `uk_username` (`username`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

                // Settings table
                "CREATE TABLE IF NOT EXISTS `settings` (
                    `key` varchar(100) NOT NULL,
                    `value` text NOT NULL,
                    `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
                    PRIMARY KEY (`key`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

                // Shortlinks table
                "CREATE TABLE IF NOT EXISTS `shortlinks` (
                    `code` varchar(100) NOT NULL,
                    `domain` varchar(255) DEFAULT NULL,
                    `og_title` varchar(500) NOT NULL,
                    `og_description` text NOT NULL,
                    `og_image_url` varchar(1000) DEFAULT '',
                    `og_type` varchar(50) DEFAULT 'website',
                    `og_site_name` varchar(255) DEFAULT '',
                    `og_locale` varchar(10) DEFAULT 'id_ID',
                    `domain_type` enum('global','user') DEFAULT NULL,
                    `domain_id` bigint(20) unsigned DEFAULT NULL,
                    `use_shim` tinyint(1) NOT NULL DEFAULT 0,
                    `is_active` tinyint(1) NOT NULL DEFAULT 1,
                    `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
                    `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
                    PRIMARY KEY (`code`),
                    KEY `idx_is_active` (`is_active`),
                    KEY `idx_domain_active` (`domain`,`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

                // Global domains table
                "CREATE TABLE IF NOT EXISTS `global_domains` (
                    `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                    `domain` varchar(255) NOT NULL,
                    `provider` enum('cpanel','cloudflare') NOT NULL DEFAULT 'cloudflare',
                    `is_wildcard` tinyint(1) NOT NULL DEFAULT 0,
                    `is_active` tinyint(1) NOT NULL DEFAULT 1,
                    `is_default` tinyint(1) NOT NULL DEFAULT 0,
                    `cpanel_host` varchar(255) DEFAULT NULL,
                    `cpanel_username` varchar(100) DEFAULT NULL,
                    `cpanel_token` text DEFAULT NULL,
                    `cf_zone_id` varchar(100) DEFAULT NULL,
                    `cf_api_token` text DEFAULT NULL,
                    `cf_proxied` tinyint(1) DEFAULT 0,
                    `dns_record_id` varchar(100) DEFAULT NULL,
                    `target_ip` varchar(45) DEFAULT NULL,
                    `target_cname` varchar(255) DEFAULT NULL,
                    `last_sync_at` timestamp NULL DEFAULT NULL,
                    `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
                    `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `uk_domain` (`domain`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

                // User domains table
                "CREATE TABLE IF NOT EXISTS `user_domains` (
                    `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                    `user_id` bigint(20) unsigned NOT NULL,
                    `domain` varchar(255) NOT NULL,
                    `provider` enum('cpanel','cloudflare') NOT NULL DEFAULT 'cloudflare',
                    `is_wildcard` tinyint(1) NOT NULL DEFAULT 0,
                    `is_active` tinyint(1) NOT NULL DEFAULT 1,
                    `cpanel_host` varchar(255) DEFAULT NULL,
                    `cpanel_username` varchar(100) DEFAULT NULL,
                    `cpanel_token` text DEFAULT NULL,
                    `cf_zone_id` varchar(100) DEFAULT NULL,
                    `cf_api_token` text DEFAULT NULL,
                    `cf_proxied` tinyint(1) DEFAULT 0,
                    `dns_record_id` varchar(100) DEFAULT NULL,
                    `target_ip` varchar(45) DEFAULT NULL,
                    `target_cname` varchar(255) DEFAULT NULL,
                    `last_sync_at` timestamp NULL DEFAULT NULL,
                    `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
                    `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `uk_user_domain` (`user_id`,`domain`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

                // Traffic analytics table
                "CREATE TABLE IF NOT EXISTS `traffic_analytics` (
                    `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                    `shortlink_id` bigint(20) unsigned DEFAULT NULL,
                    `shortlink_code` varchar(50) NOT NULL,
                    `target_url` text NOT NULL,
                    `redirect_url` text DEFAULT NULL,
                    `ip_bin` varbinary(16) NOT NULL,
                    `country_code` char(2) DEFAULT NULL,
                    `country_name` varchar(100) DEFAULT NULL,
                    `city` varchar(100) DEFAULT NULL,
                    `continent_code` char(2) DEFAULT NULL,
                    `latitude` decimal(10,8) DEFAULT NULL,
                    `longitude` decimal(11,8) DEFAULT NULL,
                    `asn` int(10) unsigned DEFAULT NULL,
                    `asn_org` varchar(255) DEFAULT NULL,
                    `isp` varchar(255) DEFAULT NULL,
                    `is_vpn` tinyint(1) NOT NULL DEFAULT 0,
                    `is_proxy` tinyint(1) NOT NULL DEFAULT 0,
                    `is_tor` tinyint(1) NOT NULL DEFAULT 0,
                    `is_datacenter` tinyint(1) NOT NULL DEFAULT 0,
                    `risk_score` tinyint(3) unsigned NOT NULL DEFAULT 0,
                    `blackbox_signal` enum('Y','N','E') DEFAULT NULL,
                    `device_type` enum('mobile','tablet','desktop','bot','unknown') NOT NULL DEFAULT 'unknown',
                    `device_brand` varchar(50) DEFAULT NULL,
                    `device_model` varchar(100) DEFAULT NULL,
                    `browser_name` varchar(50) DEFAULT NULL,
                    `browser_version` varchar(50) DEFAULT NULL,
                    `os_name` varchar(50) DEFAULT NULL,
                    `os_version` varchar(50) DEFAULT NULL,
                    `platform` varchar(50) DEFAULT NULL,
                    `user_agent` text DEFAULT NULL,
                    `is_bot` tinyint(1) NOT NULL DEFAULT 0,
                    `bot_name` varchar(100) DEFAULT NULL,
                    `referer` text DEFAULT NULL,
                    `http_method` enum('GET','POST','HEAD','PUT','DELETE','OPTIONS','PATCH') NOT NULL DEFAULT 'GET',
                    `http_protocol` enum('HTTP/1.0','HTTP/1.1','HTTP/2','HTTP/3') DEFAULT NULL,
                    `visited_at` datetime(6) NOT NULL DEFAULT current_timestamp(6),
                    `response_time_ms` int(10) unsigned DEFAULT NULL,
                    PRIMARY KEY (`id`),
                    KEY `idx_shortlink_code_visited` (`shortlink_code`,`visited_at`),
                    KEY `idx_visited_at` (`visited_at`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

                // Traffic summary daily table
                "CREATE TABLE IF NOT EXISTS `traffic_summary_daily` (
                    `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                    `shortlink_code` varchar(50) NOT NULL,
                    `summary_date` date NOT NULL,
                    `total_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `unique_ips` int(10) unsigned NOT NULL DEFAULT 0,
                    `bot_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `mobile_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `desktop_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `tablet_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `risky_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `vpn_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `proxy_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `tor_visits` int(10) unsigned NOT NULL DEFAULT 0,
                    `top_countries` longtext DEFAULT NULL,
                    `top_devices` longtext DEFAULT NULL,
                    `top_browsers` longtext DEFAULT NULL,
                    `top_referrers` longtext DEFAULT NULL,
                    `avg_response_time_ms` int(10) unsigned DEFAULT NULL,
                    `created_at` datetime(6) NOT NULL DEFAULT current_timestamp(6),
                    `updated_at` datetime(6) NOT NULL DEFAULT current_timestamp(6) ON UPDATE current_timestamp(6),
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `uk_shortlink_date` (`shortlink_code`,`summary_date`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

                // Default settings
                "INSERT IGNORE INTO `settings` (`key`, `value`) VALUES
                    ('default_locale', 'id_ID'),
                    ('default_og_type', 'website'),
                    ('default_site_name', 'OG Shortlink'),
                    ('redirect_url', '')",
            ];

            foreach ($tables as $sql) {
                try {
                    $pdo->exec($sql);
                } catch (PDOException $e) {
                    // Ignore duplicate errors
                    if (strpos($e->getMessage(), 'Duplicate') === false &&
                        strpos($e->getMessage(), 'already exists') === false) {
                        throw $e;
                    }
                }
            }

            // 3. Create admin user
            $adminUser = trim($_POST['admin_username']);
            $adminPass = $_POST['admin_password'];
            $adminName = trim($_POST['admin_name']) ?: 'Administrator';

            if (strlen($adminPass) < 8) {
                throw new Exception('Password must be at least 8 characters');
            }

            $hash = password_hash($adminPass, PASSWORD_BCRYPT, ['cost' => 12]);

            // Check if user exists
            $stmt = $pdo->prepare('SELECT id FROM users WHERE username = ?');
            $stmt->execute([$adminUser]);
            if (!$stmt->fetch()) {
                $stmt = $pdo->prepare('INSERT INTO users (username, password_hash, name, role, is_active) VALUES (?, ?, ?, ?, 1)');
                $stmt->execute([$adminUser, $hash, $adminName, 'admin']);
            }

            // 4. Generate .env file
            $sessionSecret = bin2hex(random_bytes(32));
            $encryptionKey = bin2hex(random_bytes(32));
            $apiKey = bin2hex(random_bytes(24));

            $baseUrl = rtrim($_POST['base_url'], '/');

            $envContent = <<<ENV
# OG Shortlink Configuration
# Generated on: {$_POST['_timestamp']}

DB_DSN="mysql:host={$_POST['db_host']};port={$_POST['db_port']};dbname={$_POST['db_name']};charset=utf8mb4"
DB_USER="{$_POST['db_user']}"
DB_PASS="{$_POST['db_pass']}"

APP_ENV="production"
APP_DEBUG="false"
PUBLIC_BASE_URL="{$baseUrl}"

SESSION_SECRET="{$sessionSecret}"
ENCRYPTION_KEY="{$encryptionKey}"

SHORTLINK_API_KEY="{$apiKey}"
SHORTLINK_HTTPS_ONLY="true"
SHORTLINK_BLOCK_IP_LITERAL="true"

TRAFFIC_LOG_ENABLED="true"
TRAFFIC_LOG_ASYNC="true"

IP_INTEL_ENABLED="false"
SHIM_ENABLED="false"
ENV;

            file_put_contents($basePath . '/.env', $envContent);
            chmod($basePath . '/.env', 0600);

            // 5. Create data directories
            $dirs = ['data', 'data/cache', 'data/logs'];
            foreach ($dirs as $dir) {
                $path = $basePath . '/' . $dir;
                if (!is_dir($path)) {
                    mkdir($path, 0755, true);
                }
            }

            // 6. Create lock file
            file_put_contents($basePath . '/data/.installed', date('c'));

            $_SESSION['installed'] = true;
            $_SESSION['base_url'] = $baseUrl;
            $step = 2;

        } catch (Exception $e) {
            $error = $e->getMessage();
        }
    }
}

// Detect base URL
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'] ?? 'localhost';
$baseUrl = "{$protocol}://{$host}";

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OG Shortlink - Quick Setup</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; padding: 2rem; }
        .container { max-width: 600px; margin: 0 auto; }
        .card { background: white; border-radius: 1rem; box-shadow: 0 25px 50px -12px rgba(0,0,0,0.25); overflow: hidden; }
        .header { background: #111827; color: white; padding: 2rem; text-align: center; }
        .header h1 { font-size: 1.5rem; margin-bottom: 0.5rem; }
        .header p { color: #9ca3af; font-size: 0.9rem; }
        .body { padding: 2rem; }
        .form-group { margin-bottom: 1.25rem; }
        .form-label { display: block; font-weight: 500; color: #374151; margin-bottom: 0.5rem; font-size: 0.875rem; }
        .form-input { width: 100%; padding: 0.75rem 1rem; border: 1px solid #d1d5db; border-radius: 0.5rem; font-size: 1rem; }
        .form-input:focus { outline: none; border-color: #4f46e5; box-shadow: 0 0 0 3px rgba(79,70,229,0.1); }
        .form-hint { font-size: 0.75rem; color: #6b7280; margin-top: 0.25rem; }
        .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; }
        .btn { display: inline-block; padding: 0.75rem 1.5rem; border-radius: 0.5rem; font-weight: 600; cursor: pointer; border: none; font-size: 0.95rem; }
        .btn-primary { background: #4f46e5; color: white; }
        .btn-primary:hover { background: #4338ca; }
        .btn-secondary { background: #f3f4f6; color: #374151; }
        .btn-block { width: 100%; }
        .alert { padding: 1rem; border-radius: 0.5rem; margin-bottom: 1.5rem; }
        .alert-error { background: #fef2f2; color: #991b1b; border: 1px solid #fecaca; }
        .alert-success { background: #ecfdf5; color: #065f46; border: 1px solid #a7f3d0; }
        .section-title { font-size: 1.1rem; font-weight: 600; color: #111827; margin-bottom: 1rem; padding-bottom: 0.5rem; border-bottom: 2px solid #f3f4f6; }
        .complete-icon { width: 80px; height: 80px; background: #10b981; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1.5rem; font-size: 2.5rem; color: white; }
        .link-list { list-style: none; }
        .link-list li { padding: 0.5rem 0; }
        .link-list a { color: #4f46e5; text-decoration: none; font-weight: 500; }
        .test-result { margin-top: 0.5rem; padding: 0.5rem; border-radius: 0.25rem; font-size: 0.875rem; }
        .test-result.success { background: #ecfdf5; color: #065f46; }
        .test-result.error { background: #fef2f2; color: #991b1b; }
        @media (max-width: 640px) { .form-row { grid-template-columns: 1fr; } }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <div class="header">
                <h1>OG Shortlink - Quick Setup</h1>
                <p>Production Installation</p>
            </div>
            <div class="body">
                <?php if ($error): ?>
                <div class="alert alert-error"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>

                <?php if ($step === 1): ?>
                <!-- STEP 1: Installation Form -->
                <form method="post" id="install-form">
                    <input type="hidden" name="action" value="install">
                    <input type="hidden" name="_timestamp" value="<?= date('Y-m-d H:i:s') ?>">

                    <h3 class="section-title">Database Configuration</h3>

                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Database Host</label>
                            <input type="text" class="form-input" name="db_host" value="localhost" required>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Port</label>
                            <input type="number" class="form-input" name="db_port" value="3306" required>
                        </div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Database Name</label>
                        <input type="text" class="form-input" name="db_name" required placeholder="cpanel_dbname">
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Database User</label>
                            <input type="text" class="form-input" name="db_user" required placeholder="cpanel_user">
                        </div>
                        <div class="form-group">
                            <label class="form-label">Database Password</label>
                            <input type="password" class="form-input" name="db_pass" required>
                        </div>
                    </div>

                    <button type="button" class="btn btn-secondary" onclick="testConnection()">Test Connection</button>
                    <div id="test-result"></div>

                    <h3 class="section-title" style="margin-top: 2rem;">Site Settings</h3>

                    <div class="form-group">
                        <label class="form-label">Site URL</label>
                        <input type="url" class="form-input" name="base_url" value="<?= htmlspecialchars($baseUrl) ?>" required>
                        <p class="form-hint">Your shortlink domain (with https://)</p>
                    </div>

                    <h3 class="section-title">Admin Account</h3>

                    <div class="form-group">
                        <label class="form-label">Admin Username</label>
                        <input type="text" class="form-input" name="admin_username" value="admin" required pattern="[a-zA-Z0-9_]{3,50}">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Admin Name</label>
                        <input type="text" class="form-input" name="admin_name" value="Administrator">
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Password</label>
                            <input type="password" class="form-input" name="admin_password" required minlength="8">
                        </div>
                        <div class="form-group">
                            <label class="form-label">Confirm Password</label>
                            <input type="password" class="form-input" name="admin_password_confirm" required minlength="8">
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block" style="margin-top: 1.5rem;">Install Now</button>
                </form>

                <?php else: ?>
                <!-- STEP 2: Complete -->
                <div style="text-align: center;">
                    <div class="complete-icon">✓</div>
                    <h2 style="margin-bottom: 0.5rem;">Installation Complete!</h2>
                    <p style="color: #6b7280; margin-bottom: 2rem;">OG Shortlink is ready to use.</p>
                </div>

                <div class="alert alert-success">
                    Your shortlink service has been installed successfully.
                </div>

                <h3 class="section-title">Next Steps</h3>
                <ul class="link-list">
                    <li><a href="<?= htmlspecialchars($_SESSION['base_url'] ?? $baseUrl) ?>/admin/">→ Go to Admin Panel</a></li>
                    <li><a href="<?= htmlspecialchars($_SESSION['base_url'] ?? $baseUrl) ?>/">→ View Homepage</a></li>
                </ul>

                <div class="alert alert-error" style="margin-top: 2rem;">
                    <strong>IMPORTANT:</strong> Delete this file (setup.php) now for security!
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
    function testConnection() {
        const form = document.getElementById('install-form');
        const formData = new FormData(form);
        formData.set('action', 'test_db');

        const result = document.getElementById('test-result');
        result.innerHTML = 'Testing...';
        result.className = 'test-result';

        fetch('', { method: 'POST', body: formData })
            .then(r => r.json())
            .then(data => {
                result.className = 'test-result ' + (data.success ? 'success' : 'error');
                result.innerHTML = (data.success ? '✓ ' : '✗ ') + data.message;
            })
            .catch(err => {
                result.className = 'test-result error';
                result.innerHTML = '✗ Connection error';
            });
    }

    document.getElementById('install-form')?.addEventListener('submit', function(e) {
        const pass = this.admin_password.value;
        const confirm = this.admin_password_confirm.value;
        if (pass !== confirm) {
            e.preventDefault();
            alert('Passwords do not match!');
        }
    });
    </script>
</body>
</html>
