<?php

declare(strict_types=1);

namespace App\Application\Shortlink;

use App\Cache;
use App\Db;
use App\ShortlinkRepository;

/**
 * Service for resolving shortlink codes to their data
 */
final class ShortlinkResolver
{
    private const CACHE_TTL = 300; // 5 minutes

    /**
     * Resolve a shortlink code to its data
     *
     * @param string $code Shortlink code
     * @param string $clientIp Client IP for cache key
     * @return array{
     *   code: string,
     *   og_title: string,
     *   og_description: string,
     *   og_image_url: string,
     *   og_type: string,
     *   og_site_name: string,
     *   og_locale: string,
     *   use_shim: int
     * }|null
     */
    public function resolve(string $code, string $clientIp): ?array
    {
        // Try cache first
        $cacheKey = Cache::shortlinkKey($clientIp, $code);
        $row = Cache::get($cacheKey);

        if ($row !== null) {
            return $row;
        }

        // Cache miss - query database
        $pdo = Db::pdo();
        $repo = new ShortlinkRepository($pdo);

        $row = $repo->findActive($code);
        if ($row === null) {
            return null;
        }

        // Store in cache
        Cache::set($cacheKey, $row, self::CACHE_TTL);

        return $row;
    }

    /**
     * Invalidate cache for a shortlink
     */
    public function invalidateCache(string $code, string $clientIp): void
    {
        $cacheKey = Cache::shortlinkKey($clientIp, $code);
        Cache::delete($cacheKey);
    }
}
