<?php

declare(strict_types=1);

namespace App\Infrastructure\Cache;

use RuntimeException;

/**
 * APCu cache implementation
 */
final class ApcuCache implements CacheInterface
{
    private string $prefix;

    public function __construct(string $prefix = 'app:')
    {
        if (!extension_loaded('apcu')) {
            throw new RuntimeException('APCu extension is not loaded');
        }

        if (!apcu_enabled()) {
            throw new RuntimeException('APCu is not enabled');
        }

        $this->prefix = $prefix;
    }

    public function get(string $key): mixed
    {
        $success = false;
        $value = apcu_fetch($this->prefix . $key, $success);

        return $success ? $value : null;
    }

    public function set(string $key, mixed $value, int $ttl = 300): bool
    {
        return apcu_store($this->prefix . $key, $value, $ttl);
    }

    public function delete(string $key): bool
    {
        return apcu_delete($this->prefix . $key);
    }

    public function has(string $key): bool
    {
        return apcu_exists($this->prefix . $key);
    }

    public function clear(): bool
    {
        return apcu_clear_cache();
    }

    public function getMultiple(array $keys): array
    {
        $prefixedKeys = array_map(fn($k) => $this->prefix . $k, $keys);
        $values = apcu_fetch($prefixedKeys);

        if (!is_array($values)) {
            return array_fill_keys($keys, null);
        }

        $result = [];
        foreach ($keys as $key) {
            $prefixedKey = $this->prefix . $key;
            $result[$key] = $values[$prefixedKey] ?? null;
        }

        return $result;
    }

    public function setMultiple(array $values, int $ttl = 300): bool
    {
        $prefixedValues = [];
        foreach ($values as $key => $value) {
            $prefixedValues[$this->prefix . $key] = $value;
        }

        $result = apcu_store($prefixedValues, null, $ttl);

        // apcu_store returns array of failed keys when storing multiple
        return is_array($result) ? empty($result) : (bool) $result;
    }
}
