<?php

declare(strict_types=1);

namespace App\Infrastructure\Cache;

/**
 * Cache interface for dependency injection and testability
 */
interface CacheInterface
{
    /**
     * Get a value from cache
     *
     * @param string $key Cache key
     * @return mixed|null Cached value or null if not found
     */
    public function get(string $key): mixed;

    /**
     * Store a value in cache
     *
     * @param string $key Cache key
     * @param mixed $value Value to cache
     * @param int $ttl Time to live in seconds (default 300 = 5 minutes)
     * @return bool Success
     */
    public function set(string $key, mixed $value, int $ttl = 300): bool;

    /**
     * Delete a value from cache
     *
     * @param string $key Cache key
     * @return bool Success
     */
    public function delete(string $key): bool;

    /**
     * Check if a key exists in cache
     *
     * @param string $key Cache key
     * @return bool
     */
    public function has(string $key): bool;

    /**
     * Clear all cached values
     *
     * @return bool Success
     */
    public function clear(): bool;

    /**
     * Get multiple values from cache
     *
     * @param array<string> $keys Cache keys
     * @return array<string, mixed> Key-value pairs (missing keys have null values)
     */
    public function getMultiple(array $keys): array;

    /**
     * Store multiple values in cache
     *
     * @param array<string, mixed> $values Key-value pairs
     * @param int $ttl Time to live in seconds
     * @return bool Success
     */
    public function setMultiple(array $values, int $ttl = 300): bool;
}
