<?php

declare(strict_types=1);

namespace App\Infrastructure\Cache;

use RuntimeException;

/**
 * File-based cache implementation (fallback when APCu not available)
 */
final class FileCache implements CacheInterface
{
    private string $directory;
    private string $prefix;

    public function __construct(string $directory, string $prefix = 'app_')
    {
        $this->directory = rtrim($directory, '/\\');
        $this->prefix = $prefix;

        if (!is_dir($this->directory)) {
            if (!mkdir($this->directory, 0700, true)) {
                throw new RuntimeException("Cannot create cache directory: {$this->directory}");
            }
        }

        if (!is_writable($this->directory)) {
            throw new RuntimeException("Cache directory not writable: {$this->directory}");
        }
    }

    public function get(string $key): mixed
    {
        $file = $this->getFilePath($key);

        if (!file_exists($file)) {
            return null;
        }

        $content = file_get_contents($file);
        if ($content === false) {
            return null;
        }

        $data = unserialize($content);
        if (!is_array($data) || !isset($data['expires'], $data['value'])) {
            $this->delete($key);
            return null;
        }

        if ($data['expires'] !== 0 && $data['expires'] < time()) {
            $this->delete($key);
            return null;
        }

        return $data['value'];
    }

    public function set(string $key, mixed $value, int $ttl = 300): bool
    {
        $file = $this->getFilePath($key);
        $data = [
            'expires' => $ttl > 0 ? time() + $ttl : 0,
            'value' => $value,
        ];

        $content = serialize($data);
        $result = file_put_contents($file, $content, LOCK_EX);

        return $result !== false;
    }

    public function delete(string $key): bool
    {
        $file = $this->getFilePath($key);

        if (!file_exists($file)) {
            return true;
        }

        return unlink($file);
    }

    public function has(string $key): bool
    {
        return $this->get($key) !== null;
    }

    public function clear(): bool
    {
        $pattern = $this->directory . '/' . $this->prefix . '*.cache';
        $files = glob($pattern);

        if ($files === false) {
            return false;
        }

        foreach ($files as $file) {
            if (!unlink($file)) {
                return false;
            }
        }

        return true;
    }

    public function getMultiple(array $keys): array
    {
        $result = [];
        foreach ($keys as $key) {
            $result[$key] = $this->get($key);
        }
        return $result;
    }

    public function setMultiple(array $values, int $ttl = 300): bool
    {
        foreach ($values as $key => $value) {
            if (!$this->set($key, $value, $ttl)) {
                return false;
            }
        }
        return true;
    }

    private function getFilePath(string $key): string
    {
        $safeKey = preg_replace('/[^a-zA-Z0-9_-]/', '_', $key);
        return $this->directory . '/' . $this->prefix . $safeKey . '.cache';
    }
}
