<?php

declare(strict_types=1);

namespace App\Infrastructure\Http;

/**
 * JSON Response for API endpoints
 */
final class JsonResponse extends Response
{
    /**
     * @param array<string, mixed> $data Data to encode as JSON
     * @param int $statusCode HTTP status code
     * @param array<string, string> $headers Additional headers
     */
    public function __construct(
        array $data,
        int $statusCode = 200,
        array $headers = []
    ) {
        $headers['Content-Type'] = 'application/json; charset=utf-8';

        $body = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR);

        parent::__construct($body, $statusCode, $headers);
    }

    /**
     * Create a success response
     *
     * @param array<string, mixed> $data Additional data
     */
    public static function success(array $data = []): self
    {
        return new self(array_merge(['ok' => true], $data));
    }

    /**
     * Create an error response
     *
     * @param string $message Error message
     * @param int $statusCode HTTP status code
     * @param array<string, mixed> $extra Additional data
     */
    public static function error(string $message, int $statusCode = 400, array $extra = []): self
    {
        return new self(
            array_merge(['ok' => false, 'error' => $message], $extra),
            $statusCode
        );
    }

    /**
     * Create a validation error response
     *
     * @param array<string, string> $errors Field => error message pairs
     */
    public static function validationError(array $errors): self
    {
        return new self(
            ['ok' => false, 'error' => 'Validation failed', 'errors' => $errors],
            422
        );
    }
}
