<?php

declare(strict_types=1);

namespace App\Infrastructure\Http;

/**
 * HTTP Redirect Response
 */
final class RedirectResponse extends Response
{
    private string $url;

    /**
     * @param string $url Target URL
     * @param int $statusCode HTTP status code (301, 302, 303, 307, 308)
     * @param array<string, string> $headers Additional headers
     */
    public function __construct(
        string $url,
        int $statusCode = 302,
        array $headers = []
    ) {
        $this->url = $url;
        $headers['Location'] = $url;

        parent::__construct('', $statusCode, $headers);
    }

    public function getUrl(): string
    {
        return $this->url;
    }

    /**
     * Create a temporary redirect (302)
     */
    public static function temporary(string $url): self
    {
        return new self($url, 302);
    }

    /**
     * Create a permanent redirect (301)
     */
    public static function permanent(string $url): self
    {
        return new self($url, 301);
    }

    /**
     * Send the redirect response
     */
    public function send(): void
    {
        http_response_code($this->statusCode);

        foreach ($this->headers as $name => $value) {
            header("{$name}: {$value}", true);
        }

        exit;
    }
}
