<?php

declare(strict_types=1);

namespace App\Infrastructure\Http;

/**
 * HTTP Request abstraction for testability
 */
final class Request
{
    /** @var array<string, string> */
    private array $query;

    /** @var array<string, mixed> */
    private array $post;

    /** @var array<string, string> */
    private array $server;

    /** @var array<string, string> */
    private array $cookies;

    private string $body;

    /**
     * @param array<string, string> $query GET parameters
     * @param array<string, mixed> $post POST parameters
     * @param array<string, string> $server Server variables
     * @param array<string, string> $cookies Cookies
     * @param string $body Raw request body
     */
    public function __construct(
        array $query = [],
        array $post = [],
        array $server = [],
        array $cookies = [],
        string $body = ''
    ) {
        $this->query = $query;
        $this->post = $post;
        $this->server = $server;
        $this->cookies = $cookies;
        $this->body = $body;
    }

    /**
     * Create request from PHP globals
     */
    public static function fromGlobals(): self
    {
        $body = file_get_contents('php://input');

        return new self(
            $_GET,
            $_POST,
            $_SERVER,
            $_COOKIE,
            is_string($body) ? $body : ''
        );
    }

    public function getMethod(): string
    {
        return strtoupper($this->server['REQUEST_METHOD'] ?? 'GET');
    }

    public function isPost(): bool
    {
        return $this->getMethod() === 'POST';
    }

    public function isGet(): bool
    {
        return $this->getMethod() === 'GET';
    }

    public function isAjax(): bool
    {
        $xhr = $this->server['HTTP_X_REQUESTED_WITH'] ?? '';
        return strtolower($xhr) === 'xmlhttprequest';
    }

    public function getQuery(string $key, string $default = ''): string
    {
        return (string) ($this->query[$key] ?? $default);
    }

    public function getPost(string $key, mixed $default = null): mixed
    {
        return $this->post[$key] ?? $default;
    }

    public function getHeader(string $name): ?string
    {
        // Convert header name to SERVER key format
        $key = 'HTTP_' . strtoupper(str_replace('-', '_', $name));
        $value = $this->server[$key] ?? null;

        return is_string($value) ? $value : null;
    }

    public function getClientIp(): string
    {
        // Check Cloudflare header first
        $cfIp = $this->server['HTTP_CF_CONNECTING_IP'] ?? null;
        if (is_string($cfIp) && $cfIp !== '' && filter_var($cfIp, FILTER_VALIDATE_IP)) {
            return $cfIp;
        }

        // Check X-Forwarded-For (take first IP only)
        $xff = $this->server['HTTP_X_FORWARDED_FOR'] ?? null;
        if (is_string($xff) && $xff !== '') {
            $ips = explode(',', $xff);
            $firstIp = trim($ips[0]);
            if (filter_var($firstIp, FILTER_VALIDATE_IP)) {
                return $firstIp;
            }
        }

        // Fallback to REMOTE_ADDR
        $remoteAddr = $this->server['REMOTE_ADDR'] ?? '0.0.0.0';
        return is_string($remoteAddr) ? $remoteAddr : '0.0.0.0';
    }

    public function getUserAgent(): string
    {
        return (string) ($this->server['HTTP_USER_AGENT'] ?? '');
    }

    public function getReferer(): string
    {
        return (string) ($this->server['HTTP_REFERER'] ?? '');
    }

    public function getBody(): string
    {
        return $this->body;
    }

    /**
     * Get JSON decoded body
     *
     * @return array<string, mixed>|null
     */
    public function getJsonBody(): ?array
    {
        if ($this->body === '') {
            return null;
        }

        $data = json_decode($this->body, true);
        return is_array($data) ? $data : null;
    }

    public function getCookie(string $name, string $default = ''): string
    {
        return (string) ($this->cookies[$name] ?? $default);
    }

    public function getUri(): string
    {
        return (string) ($this->server['REQUEST_URI'] ?? '/');
    }

    public function getPath(): string
    {
        $uri = $this->getUri();
        $pos = strpos($uri, '?');

        return $pos !== false ? substr($uri, 0, $pos) : $uri;
    }

    public function isSecure(): bool
    {
        $https = $this->server['HTTPS'] ?? '';
        return !empty($https) && $https !== 'off';
    }

    public function getHost(): string
    {
        return (string) ($this->server['HTTP_HOST'] ?? 'localhost');
    }

    /**
     * Get all query parameters
     *
     * @return array<string, string>
     */
    public function getAllQuery(): array
    {
        return $this->query;
    }

    /**
     * Get all post parameters
     *
     * @return array<string, mixed>
     */
    public function getAllPost(): array
    {
        return $this->post;
    }
}
