<?php

declare(strict_types=1);

namespace App\Infrastructure\Persistence;

use PDO;

/**
 * Abstract domain repository with shared functionality
 *
 * Contains common logic for GlobalDomainRepository and UserDomainRepository
 */
abstract class AbstractDomainRepository
{
    /**
     * Table name for this repository
     */
    protected string $tableName;

    /**
     * Allowed fields for dynamic update
     *
     * @var list<string>
     */
    protected array $allowedUpdateFields = [
        'domain',
        'provider',
        'is_wildcard',
        'is_active',
        'cpanel_host',
        'cpanel_username',
        'cpanel_token',
        'cf_zone_id',
        'cf_api_token',
        'cf_proxied',
        'dns_record_id',
        'target_ip',
        'target_cname',
        'last_sync_at',
    ];

    public function __construct(protected readonly PDO $pdo)
    {
    }

    /**
     * Build dynamic update query with whitelisted fields
     *
     * @param array<string, mixed> $data
     * @param array<string, mixed> $baseParams Parameters already set (id, user_id, etc)
     * @return array{sql: string, params: array<string, mixed>}|null Returns null if no valid fields
     */
    protected function buildUpdateQuery(array $data, array $baseParams): ?array
    {
        $updates = [];
        $params = $baseParams;

        foreach ($data as $key => $value) {
            if (in_array($key, $this->allowedUpdateFields, true)) {
                $updates[] = $key . ' = :' . $key;
                $params[$key] = $value;
            }
        }

        if (empty($updates)) {
            return null;
        }

        return [
            'sql' => implode(', ', $updates),
            'params' => $params,
        ];
    }

    /**
     * Execute insert and return last insert ID
     *
     * @param string $sql
     * @param array<string, mixed> $params
     * @return int
     */
    protected function executeInsert(string $sql, array $params): int
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        return (int) $this->pdo->lastInsertId();
    }

    /**
     * Execute update and return affected rows > 0
     *
     * @param string $sql
     * @param array<string, mixed> $params
     * @return bool
     */
    protected function executeUpdate(string $sql, array $params): bool
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        return $stmt->rowCount() > 0;
    }

    /**
     * Execute delete and return success status
     *
     * @param string $sql
     * @param array<string, mixed> $params
     * @return bool
     */
    protected function executeDelete(string $sql, array $params): bool
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        return $stmt->rowCount() > 0;
    }

    /**
     * Fetch single row
     *
     * @param string $sql
     * @param array<string, mixed> $params
     * @return array<string, mixed>|null
     */
    protected function fetchOne(string $sql, array $params): ?array
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        $row = $stmt->fetch();
        return is_array($row) ? $row : null;
    }

    /**
     * Fetch all rows
     *
     * @param string $sql
     * @param array<string, mixed> $params
     * @return list<array<string, mixed>>
     */
    protected function fetchAll(string $sql, array $params = []): array
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        $rows = $stmt->fetchAll();
        return is_array($rows) ? $rows : [];
    }

    /**
     * Check if row exists
     *
     * @param string $sql
     * @param array<string, mixed> $params
     * @return bool
     */
    protected function rowExists(string $sql, array $params): bool
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchColumn() !== false;
    }

    /**
     * Get common insert fields (shared between global and user domains)
     *
     * @return list<string>
     */
    protected function getCommonInsertFields(): array
    {
        return [
            'domain',
            'provider',
            'is_wildcard',
            'cpanel_host',
            'cpanel_username',
            'cpanel_token',
            'cf_zone_id',
            'cf_api_token',
            'cf_proxied',
            'target_ip',
            'target_cname',
        ];
    }

    /**
     * Get common insert params with null defaults
     *
     * @param array<string, mixed> $data
     * @return array<string, mixed>
     */
    protected function getCommonInsertParams(array $data): array
    {
        return [
            'domain' => $data['domain'],
            'provider' => $data['provider'],
            'is_wildcard' => $data['is_wildcard'] ?? 0,
            'cpanel_host' => $data['cpanel_host'] ?? null,
            'cpanel_username' => $data['cpanel_username'] ?? null,
            'cpanel_token' => $data['cpanel_token'] ?? null,
            'cf_zone_id' => $data['cf_zone_id'] ?? null,
            'cf_api_token' => $data['cf_api_token'] ?? null,
            'cf_proxied' => $data['cf_proxied'] ?? 0,
            'target_ip' => $data['target_ip'] ?? null,
            'target_cname' => $data['target_cname'] ?? null,
        ];
    }
}
