<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Analytics\TrafficLogger;
use App\Http;
use App\Infrastructure\Http\JsonResponse;
use PDO;

/**
 * API controller for analytics
 */
final class AnalyticsApiController extends BaseApiController
{
    /**
     * Get analytics summary for a shortlink
     *
     * @return JsonResponse
     */
    public function getSummary(): JsonResponse
    {
        $code = (string) ($_GET['code'] ?? '');
        $days = (int) ($_GET['days'] ?? 30);

        if ($code === '') {
            return $this->error('Shortlink code required');
        }

        $code = Http::validateCode($code);
        if ($code === '') {
            return $this->error('Invalid code');
        }

        try {
            $logger = new TrafficLogger($this->pdo);
            $summary = $logger->getSummary($code, $days);

            return $this->success(['summary' => $summary]);
        } catch (\Throwable $e) {
            error_log('analytics_summary_error: ' . $e->getMessage());
            return $this->error('Failed to fetch analytics');
        }
    }

    /**
     * Get recent visitors for a shortlink
     *
     * @return JsonResponse
     */
    public function getVisitors(): JsonResponse
    {
        $code = (string) ($_GET['code'] ?? '');
        $page = max(1, (int) ($_GET['page'] ?? 1));
        $pageSize = min(100, max(1, (int) ($_GET['page_size'] ?? 50)));

        if ($code === '') {
            return $this->error('Shortlink code required');
        }

        $code = Http::validateCode($code);
        if ($code === '') {
            return $this->error('Invalid code');
        }

        try {
            $offset = ($page - 1) * $pageSize;

            // Get total count
            $stmt = $this->pdo->prepare('SELECT COUNT(*) FROM traffic_analytics WHERE shortlink_code = ?');
            $stmt->execute([$code]);
            $total = (int) $stmt->fetchColumn();

            // Get visitors
            $stmt = $this->pdo->prepare('
                SELECT
                    ip_text as ip_address,
                    country_code,
                    country_name,
                    city,
                    device_type,
                    device_brand,
                    browser_name,
                    os_name,
                    is_bot,
                    risk_score,
                    visited_at
                FROM traffic_analytics
                WHERE shortlink_code = ?
                ORDER BY visited_at DESC
                LIMIT ? OFFSET ?
            ');
            $stmt->execute([$code, $pageSize, $offset]);
            $visitors = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return $this->success(['visitors' => $visitors, 'total' => $total]);
        } catch (\Throwable $e) {
            error_log('analytics_visitors_error: ' . $e->getMessage());
            return $this->error('Failed to fetch visitors');
        }
    }

    /**
     * Get chart data (visits per day)
     *
     * @return JsonResponse
     */
    public function getChart(): JsonResponse
    {
        $code = (string) ($_GET['code'] ?? '');
        $days = (int) ($_GET['days'] ?? 30);

        if ($code === '') {
            return $this->error('Shortlink code required');
        }

        $code = Http::validateCode($code);
        if ($code === '') {
            return $this->error('Invalid code');
        }

        try {
            $stmt = $this->pdo->prepare('
                SELECT
                    DATE(visited_at) as date,
                    COUNT(*) as visits
                FROM traffic_analytics
                WHERE shortlink_code = ?
                  AND visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY DATE(visited_at)
                ORDER BY date ASC
            ');
            $stmt->execute([$code, $days]);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $dates = [];
            $visits = [];

            foreach ($rows as $row) {
                $dates[] = $row['date'];
                $visits[] = (int) $row['visits'];
            }

            return $this->success(['chart' => ['dates' => $dates, 'visits' => $visits]]);
        } catch (\Throwable $e) {
            error_log('analytics_chart_error: ' . $e->getMessage());
            return $this->error('Failed to fetch chart data');
        }
    }
}
