<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Admin\Auth;
use App\Infrastructure\Http\JsonResponse;

/**
 * API Router - routes requests to appropriate controllers
 */
final class ApiRouter
{
    /**
     * Route and handle API request
     *
     * @param string $action
     * @return JsonResponse
     */
    public function route(string $action): JsonResponse
    {
        // Rate limiting check (applied to all actions)
        $baseController = new BaseApiController();
        $rateLimitError = $this->checkRateLimit();
        if ($rateLimitError !== null) {
            return $rateLimitError;
        }

        // GET actions (no CSRF required)
        $getActions = [
            'list' => [ShortlinkApiController::class, 'list'],
            'get' => [ShortlinkApiController::class, 'get'],
            'get_settings' => [SettingsApiController::class, 'get'],
            'get_public_settings' => [SettingsApiController::class, 'getPublic'],
            'get_security_settings' => [SettingsApiController::class, 'getSecurity', true],
            'list_users' => [UserApiController::class, 'list', true],
            'get_user' => [UserApiController::class, 'get', true],
            'get_facebook_protection_config' => [ProtectionApiController::class, 'getFacebookConfig', true],
            'get_protection_metrics' => [ProtectionApiController::class, 'getMetrics', true],
            'get_shim_config' => [ProtectionApiController::class, 'getShimConfig', true],
            'check_domain_health' => [ProtectionApiController::class, 'checkDomainHealth', true],
            'get_env_var' => [EnvApiController::class, 'get', true],
            'analytics_summary' => [AnalyticsApiController::class, 'getSummary'],
            'analytics_visitors' => [AnalyticsApiController::class, 'getVisitors'],
            'analytics_chart' => [AnalyticsApiController::class, 'getChart'],
            'list_user_domains' => [DomainApiController::class, 'listUserDomains'],
            'list_global_domains' => [DomainApiController::class, 'listGlobalDomains', true],
            'list_available_domains' => [DomainApiController::class, 'listAvailable'],
        ];

        // POST actions (CSRF required)
        $postActions = [
            'create' => [ShortlinkApiController::class, 'create'],
            'update' => [ShortlinkApiController::class, 'update'],
            'set_active' => [ShortlinkApiController::class, 'setActive'],
            'delete' => [ShortlinkApiController::class, 'delete'],
            'bulk_delete' => [ShortlinkApiController::class, 'bulkDelete'],
            'create_third_party_shortlink' => [ThirdPartyApiController::class, 'createShortlink'],
            'create_user' => [UserApiController::class, 'create', true],
            'update_user' => [UserApiController::class, 'update', true],
            'delete_user' => [UserApiController::class, 'delete', true],
            'bulk_delete_users' => [UserApiController::class, 'bulkDelete', true],
            'toggle_user_active' => [UserApiController::class, 'toggleActive', true],
            'update_security_settings' => [SettingsApiController::class, 'updateSecurity', true],
            'create_user_domain' => [DomainApiController::class, 'createUserDomain'],
            'sync_user_domain_dns' => [DomainApiController::class, 'syncUserDomainDns'],
            'delete_user_domain' => [DomainApiController::class, 'deleteUserDomain'],
            'create_global_domain' => [DomainApiController::class, 'createGlobalDomain', true],
            'sync_global_domain_dns' => [DomainApiController::class, 'syncGlobalDomainDns', true],
            'delete_global_domain' => [DomainApiController::class, 'deleteGlobalDomain', true],
            'update_facebook_protection_config' => [ProtectionApiController::class, 'updateFacebookConfig', true],
            'update_env_vars' => [EnvApiController::class, 'update', true],
        ];

        // Check GET actions first
        if (isset($getActions[$action])) {
            return $this->handleAction($getActions[$action], false);
        }

        // POST-only actions require POST method
        if (isset($postActions[$action])) {
            if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
                return JsonResponse::error('Method not allowed', 405);
            }

            // Verify CSRF
            $csrf = (string) ($_SERVER['HTTP_X_CSRF_TOKEN'] ?? '');
            if (!Auth::verifyCsrfToken($csrf)) {
                return JsonResponse::error('CSRF invalid', 400);
            }

            return $this->handleAction($postActions[$action], true);
        }

        return JsonResponse::error('Unknown action', 400);
    }

    /**
     * Handle action with controller
     *
     * @param array{0: class-string, 1: string, 2?: bool} $handler
     * @param bool $isPost
     * @return JsonResponse
     */
    private function handleAction(array $handler, bool $isPost): JsonResponse
    {
        [$controllerClass, $method] = $handler;
        $requiresAdmin = $handler[2] ?? false;

        // Check admin requirement
        if ($requiresAdmin && !Auth::isAdmin()) {
            return JsonResponse::error('Admin role required', 403);
        }

        // Create controller and call method
        /** @var BaseApiController $controller */
        $controller = new $controllerClass();
        return $controller->$method();
    }

    /**
     * Check rate limiting
     *
     * @return JsonResponse|null
     */
    private function checkRateLimit(): ?JsonResponse
    {
        $controller = new class extends BaseApiController {
            public function doRateLimitCheck(): ?JsonResponse
            {
                return $this->checkRateLimit();
            }
        };

        return $controller->doRateLimitCheck();
    }
}
