<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Admin\Auth;
use App\Db;
use App\Infrastructure\Http\JsonResponse;
use App\RateLimiter;
use App\SecurityConfig;
use PDO;

/**
 * Base API controller with shared functionality
 */
abstract class BaseApiController
{
    protected PDO $pdo;

    public function __construct(?PDO $pdo = null)
    {
        $this->pdo = $pdo ?? Db::pdo();
    }

    /**
     * Read JSON body from request
     *
     * @return array<string, mixed>
     */
    protected function readJson(): array
    {
        $raw = file_get_contents('php://input');
        $raw = is_string($raw) ? $raw : '';

        $data = json_decode($raw, true);
        return is_array($data) ? $data : [];
    }

    /**
     * Check rate limiting
     *
     * @return JsonResponse|null Returns error response if rate limited, null if OK
     */
    protected function checkRateLimit(): ?JsonResponse
    {
        if (!SecurityConfig::isRateLimitEnabled()) {
            return null;
        }

        $whitelistAdmin = SecurityConfig::isRateLimitWhitelistAdmin();
        if ($whitelistAdmin && Auth::isAdmin()) {
            return null;
        }

        $rateLimiter = new RateLimiter();
        $limitBy = SecurityConfig::getRateLimitBy();

        $identifier = '';
        if ($limitBy === 'ip' || $limitBy === 'ip_user') {
            $identifier .= $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        }
        if ($limitBy === 'user' || $limitBy === 'ip_user') {
            $userId = $_SESSION['user_id'] ?? 0;
            $identifier .= ($identifier ? '_' : '') . 'user_' . $userId;
        }

        $maxRequests = SecurityConfig::getRateLimitMaxRequests();
        $window = SecurityConfig::getRateLimitWindow();

        if (!$rateLimiter->check($identifier, $maxRequests, $window)) {
            return JsonResponse::error('Rate limit exceeded. Please try again later.', 429, [
                'retry_after' => $window,
            ]);
        }

        return null;
    }

    /**
     * Require admin role
     *
     * @return JsonResponse|null Returns error response if not admin, null if OK
     */
    protected function requireAdmin(): ?JsonResponse
    {
        if (!Auth::isAdmin()) {
            return JsonResponse::error('Admin role required', 403);
        }
        return null;
    }

    /**
     * Require POST method
     *
     * @return JsonResponse|null Returns error response if not POST, null if OK
     */
    protected function requirePost(): ?JsonResponse
    {
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            return JsonResponse::error('Method not allowed', 405);
        }
        return null;
    }

    /**
     * Verify CSRF token
     *
     * @return JsonResponse|null Returns error response if invalid, null if OK
     */
    protected function verifyCsrf(): ?JsonResponse
    {
        $csrf = (string) ($_SERVER['HTTP_X_CSRF_TOKEN'] ?? '');
        if (!Auth::verifyCsrfToken($csrf)) {
            return JsonResponse::error('CSRF invalid', 400);
        }
        return null;
    }

    /**
     * Get current user ID
     */
    protected function getUserId(): int
    {
        return Auth::getUserId();
    }

    /**
     * Create success response
     *
     * @param array<string, mixed> $data
     */
    protected function success(array $data = []): JsonResponse
    {
        return JsonResponse::success($data);
    }

    /**
     * Create error response
     *
     * @param string $message
     * @param int $statusCode
     * @param array<string, mixed> $extra
     */
    protected function error(string $message, int $statusCode = 400, array $extra = []): JsonResponse
    {
        return JsonResponse::error($message, $statusCode, $extra);
    }
}
