<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\CloudflareApi;
use App\CpanelApi;
use App\GlobalDomainRepository;
use App\Infrastructure\Http\JsonResponse;
use App\SecurityConfig;
use App\UserDomainRepository;
use RuntimeException;

/**
 * API controller for domain management
 */
final class DomainApiController extends BaseApiController
{
    /**
     * List user's domains
     *
     * @return JsonResponse
     */
    public function listUserDomains(): JsonResponse
    {
        $userId = $this->getUserId();
        $repo = new UserDomainRepository($this->pdo);
        $domains = $repo->listByUser($userId, true);

        return $this->success(['domains' => $domains]);
    }

    /**
     * List all global domains (admin only)
     *
     * @return JsonResponse
     */
    public function listGlobalDomains(): JsonResponse
    {
        $repo = new GlobalDomainRepository($this->pdo);
        $domains = $repo->listAll();

        return $this->success(['domains' => $domains]);
    }

    /**
     * List all available domains (global + user's)
     *
     * @return JsonResponse
     */
    public function listAvailable(): JsonResponse
    {
        $userId = $this->getUserId();

        $globalRepo = new GlobalDomainRepository($this->pdo);
        $userRepo = new UserDomainRepository($this->pdo);

        $global = $globalRepo->listActive();
        $user = $userRepo->listByUser($userId, true);

        return $this->success([
            'global_domains' => $global,
            'user_domains' => $user,
        ]);
    }

    /**
     * Create user domain
     *
     * @return JsonResponse
     */
    public function createUserDomain(): JsonResponse
    {
        $data = $this->readJson();
        $data['user_id'] = $this->getUserId();

        $repo = new UserDomainRepository($this->pdo);

        try {
            $id = $repo->create($data);
            $this->syncDomain('user', $id, $data);

            return $this->success(['id' => $id]);
        } catch (\Throwable $e) {
            return $this->error('Failed to create domain: ' . $e->getMessage());
        }
    }

    /**
     * Sync user domain DNS
     *
     * @return JsonResponse
     */
    public function syncUserDomainDns(): JsonResponse
    {
        $data = $this->readJson();
        $id = (int) ($data['id'] ?? 0);
        $userId = $this->getUserId();

        $repo = new UserDomainRepository($this->pdo);
        $domain = $repo->findById($id, $userId);

        if (!$domain) {
            return $this->error('Domain not found', 404);
        }

        try {
            $this->syncDomain('user', $id, $domain);
            return $this->success();
        } catch (\Throwable $e) {
            return $this->error('DNS sync failed: ' . $e->getMessage());
        }
    }

    /**
     * Delete user domain
     *
     * @return JsonResponse
     */
    public function deleteUserDomain(): JsonResponse
    {
        $data = $this->readJson();
        $id = (int) ($data['id'] ?? 0);
        $userId = $this->getUserId();

        $repo = new UserDomainRepository($this->pdo);
        $success = $repo->delete($id, $userId);

        if (!$success) {
            return $this->error('Failed to delete');
        }

        return $this->success();
    }

    /**
     * Create global domain (admin only)
     *
     * @return JsonResponse
     */
    public function createGlobalDomain(): JsonResponse
    {
        $data = $this->readJson();
        $repo = new GlobalDomainRepository($this->pdo);

        try {
            $id = $repo->create($data);
            $this->syncDomain('global', $id, $data);

            return $this->success(['id' => $id]);
        } catch (\Throwable $e) {
            return $this->error('Failed to create domain: ' . $e->getMessage());
        }
    }

    /**
     * Sync global domain DNS (admin only)
     *
     * @return JsonResponse
     */
    public function syncGlobalDomainDns(): JsonResponse
    {
        $data = $this->readJson();
        $id = (int) ($data['id'] ?? 0);

        $repo = new GlobalDomainRepository($this->pdo);
        $domain = $repo->findById($id);

        if (!$domain) {
            return $this->error('Domain not found', 404);
        }

        try {
            $this->syncDomain('global', $id, $domain);
            return $this->success();
        } catch (\Throwable $e) {
            return $this->error('DNS sync failed: ' . $e->getMessage());
        }
    }

    /**
     * Delete global domain (admin only)
     *
     * @return JsonResponse
     */
    public function deleteGlobalDomain(): JsonResponse
    {
        $data = $this->readJson();
        $id = (int) ($data['id'] ?? 0);

        $repo = new GlobalDomainRepository($this->pdo);
        $success = $repo->delete($id);

        if (!$success) {
            return $this->error('Failed to delete');
        }

        return $this->success();
    }

    /**
     * Sync domain to DNS provider
     *
     * @param string $type 'global' or 'user'
     * @param int $id Domain ID
     * @param array<string, mixed> $data Domain configuration
     * @throws RuntimeException On sync failure
     */
    private function syncDomain(string $type, int $id, array $data): void
    {
        $provider = $data['provider'] ?? '';

        if ($provider === 'cloudflare') {
            $this->syncCloudflare($type, $id, $data);
        } elseif ($provider === 'cpanel') {
            $this->syncCpanel($type, $id, $data);
        } else {
            throw new RuntimeException('Invalid DNS provider: ' . $provider);
        }

        // Update last_sync_at timestamp
        $table = $type === 'global' ? 'global_domains' : 'user_domains';
        $stmt = $this->pdo->prepare("UPDATE {$table} SET last_sync_at = NOW() WHERE id = ?");
        $stmt->execute([$id]);
    }

    /**
     * Sync to Cloudflare
     *
     * @param string $type
     * @param int $id
     * @param array<string, mixed> $data
     */
    private function syncCloudflare(string $type, int $id, array $data): void
    {
        $apiToken = (string) ($data['cf_api_token'] ?? '');
        if ($apiToken === '') {
            $apiToken = SecurityConfig::getCloudflareApiToken();
        }

        if ($apiToken === '') {
            throw new RuntimeException('Cloudflare API token not configured');
        }

        $api = new CloudflareApi($apiToken);

        $zoneDomain = $this->extractRootDomain((string) $data['domain']);
        $zoneResult = $api->getZoneId($zoneDomain);

        if (!$zoneResult['success']) {
            throw new RuntimeException('Failed to get zone ID: ' . $zoneResult['message']);
        }

        $recordName = ((int) ($data['is_wildcard'] ?? 0) === 1) ? '*' : $data['domain'];
        $recordType = !empty($data['target_ip']) ? 'A' : 'CNAME';
        $recordContent = !empty($data['target_ip']) ? (string) $data['target_ip'] : (string) $data['target_cname'];

        if ($recordContent === '') {
            throw new RuntimeException('Target IP or CNAME required');
        }

        $result = $api->addDnsRecord(
            $recordType,
            $recordName,
            $recordContent,
            1,
            (bool) ($data['cf_proxied'] ?? false)
        );

        if (!$result['success']) {
            throw new RuntimeException('Failed to add DNS record: ' . $result['message']);
        }

        $recordId = (string) ($result['data']['result']['id'] ?? '');
        if ($recordId !== '') {
            $table = $type === 'global' ? 'global_domains' : 'user_domains';
            $stmt = $this->pdo->prepare("UPDATE {$table} SET dns_record_id = ?, cf_zone_id = ? WHERE id = ?");
            $stmt->execute([$recordId, $zoneResult['zone_id'], $id]);
        }
    }

    /**
     * Sync to cPanel
     *
     * @param string $type
     * @param int $id
     * @param array<string, mixed> $data
     */
    private function syncCpanel(string $type, int $id, array $data): void
    {
        $host = (string) ($data['cpanel_host'] ?? '') ?: SecurityConfig::getCpanelHost();
        $username = (string) ($data['cpanel_username'] ?? '') ?: SecurityConfig::getCpanelUsername();
        $token = (string) ($data['cpanel_token'] ?? '') ?: SecurityConfig::getCpanelToken();

        if ($host === '' || $username === '' || $token === '') {
            throw new RuntimeException('cPanel credentials not configured');
        }

        $api = new CpanelApi($host, $username, $token);

        $domain = (string) $data['domain'];
        $rootDomain = $this->extractRootDomain($domain);

        $result = $api->addSubdomain($domain, $rootDomain, 'public_html/' . $domain);

        if (!$result['success']) {
            throw new RuntimeException('Failed to add subdomain: ' . $result['message']);
        }

        if ((int) ($data['is_wildcard'] ?? 0) === 1) {
            $recordType = !empty($data['target_ip']) ? 'A' : 'CNAME';
            $recordContent = !empty($data['target_ip']) ? (string) $data['target_ip'] : (string) $data['target_cname'];

            $dnsResult = $api->addDnsRecord($rootDomain, '*.' . $domain, $recordType, $recordContent);
            if (!$dnsResult['success']) {
                error_log('Warning: Failed to add wildcard DNS record: ' . $dnsResult['message']);
            }
        }
    }

    /**
     * Extract root domain from subdomain
     *
     * @param string $domain
     * @return string
     */
    private function extractRootDomain(string $domain): string
    {
        $parts = explode('.', $domain);
        if (count($parts) <= 2) {
            return $domain;
        }

        return implode('.', array_slice($parts, -2));
    }
}
