<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Infrastructure\Http\JsonResponse;

/**
 * API controller for environment variable management (admin only)
 */
final class EnvApiController extends BaseApiController
{
    private string $envFile;

    public function __construct()
    {
        parent::__construct();
        $this->envFile = dirname(__DIR__, 4) . '/.env';
    }

    /**
     * Get environment variables
     *
     * @return JsonResponse
     */
    public function get(): JsonResponse
    {
        $keys = (string) ($_GET['keys'] ?? '');

        try {
            $keyList = array_map('trim', explode(',', $keys));
            $data = [];

            foreach ($keyList as $key) {
                if ($key !== '') {
                    $data[$key] = (string) getenv($key);
                }
            }

            return $this->success(['data' => $data]);
        } catch (\Throwable $e) {
            return $this->error($e->getMessage());
        }
    }

    /**
     * Update environment variables
     *
     * @return JsonResponse
     */
    public function update(): JsonResponse
    {
        try {
            $data = $this->readJson();

            $lines = [];
            if (file_exists($this->envFile)) {
                $lines = file($this->envFile, FILE_IGNORE_NEW_LINES);
                if ($lines === false) {
                    $lines = [];
                }
            }

            foreach ($data as $key => $value) {
                $found = false;
                foreach ($lines as $i => $line) {
                    if (str_starts_with($line, $key . '=')) {
                        $lines[$i] = $key . '=' . $value;
                        $found = true;
                        break;
                    }
                }

                if (!$found) {
                    $lines[] = $key . '=' . $value;
                }
            }

            $written = file_put_contents($this->envFile, implode("\n", $lines) . "\n");
            if ($written === false) {
                return $this->error('Failed to write .env file');
            }

            return $this->success();
        } catch (\Throwable $e) {
            return $this->error($e->getMessage());
        }
    }
}
