<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Cache;
use App\Env;
use App\FacebookProtection;
use App\Infrastructure\Http\JsonResponse;

/**
 * API controller for protection features (Facebook, Shim, etc.)
 */
final class ProtectionApiController extends BaseApiController
{
    private string $configDir;

    public function __construct()
    {
        parent::__construct();
        $this->configDir = dirname(__DIR__, 4) . '/config';
    }

    /**
     * Get Facebook protection configuration
     *
     * @return JsonResponse
     */
    public function getFacebookConfig(): JsonResponse
    {
        try {
            $configFile = $this->configDir . '/facebook_protection.php';
            if (!file_exists($configFile)) {
                return $this->success(['config' => []]);
            }

            $config = require $configFile;
            return $this->success(['config' => $config]);
        } catch (\Throwable $e) {
            return $this->error($e->getMessage());
        }
    }

    /**
     * Update Facebook protection configuration
     *
     * @return JsonResponse
     */
    public function updateFacebookConfig(): JsonResponse
    {
        try {
            $data = $this->readJson();

            $configFile = $this->configDir . '/facebook_protection.php';
            $existingConfig = file_exists($configFile) ? require $configFile : [];

            $config = array_merge($existingConfig, [
                'enabled' => $data['enabled'] ?? true,
                'rate_limit' => [
                    'enabled' => $data['rate_limit_enabled'] ?? true,
                    'window_seconds' => $data['rate_limit_window'] ?? 60,
                    'max_requests' => $data['rate_limit_max_requests'] ?? 30,
                    'suspicious_threshold' => $data['rate_limit_suspicious_threshold'] ?? 10,
                    'violation_cooldown' => $data['rate_limit_violation_cooldown'] ?? 600,
                ],
                'destination_check' => [
                    'enabled' => $data['destination_check_enabled'] ?? true,
                    'block_suspicious_tlds' => $data['destination_check_block_suspicious_tlds'] ?? true,
                    'suspicious_tlds' => $data['destination_check_suspicious_tlds'] ?? ['.tk', '.ml', '.ga', '.cf', '.gq', '.xyz', '.top'],
                    'scan_path_patterns' => $data['destination_check_scan_path_patterns'] ?? true,
                    'malicious_keywords' => $data['destination_check_malicious_keywords'] ?? ['phishing', 'scam', 'fraud', 'hack'],
                ],
                'ip_validation' => [
                    'enabled' => $data['ip_validation_enabled'] ?? true,
                    'strict_mode' => $data['ip_validation_strict_mode'] ?? false,
                    'log_suspicious' => $data['ip_validation_log_suspicious'] ?? true,
                ],
                'health_check' => [
                    'enabled' => $data['health_check_enabled'] ?? true,
                    'cache_ttl' => $data['health_check_cache_ttl'] ?? 300,
                    'ban_duration' => $data['health_check_ban_duration'] ?? 3600,
                    'cooldown_duration' => $data['health_check_cooldown_duration'] ?? 600,
                ],
                'response_headers' => [
                    'vary_by_user_agent' => $data['response_headers_vary_by_user_agent'] ?? true,
                    'allow_bot_caching' => $data['response_headers_allow_bot_caching'] ?? true,
                    'bot_cache_ttl' => $data['response_headers_bot_cache_ttl'] ?? 300,
                ],
                'logging' => [
                    'enabled' => $data['logging_enabled'] ?? true,
                    'log_violations' => $data['logging_log_violations'] ?? true,
                    'log_destination_blocks' => $data['logging_log_destination_blocks'] ?? true,
                ],
            ]);

            $configContent = "<?php\n\ndeclare(strict_types=1);\n\nreturn " . var_export($config, true) . ";\n";
            $written = file_put_contents($configFile, $configContent);

            if ($written === false) {
                return $this->error('Failed to write config file');
            }

            return $this->success();
        } catch (\Throwable $e) {
            return $this->error($e->getMessage());
        }
    }

    /**
     * Get protection metrics from cache
     *
     * @return JsonResponse
     */
    public function getMetrics(): JsonResponse
    {
        try {
            $facebook = [
                'rate_limit_exceeded' => (int) (Cache::get('fb_metric:rate_limit_exceeded') ?: 0),
                'destination_blocks' => (int) (Cache::get('fb_metric:destination_blocks') ?: 0),
                'suspicious_requests' => (int) (Cache::get('fb_metric:suspicious_requests') ?: 0),
                'violations' => (int) (Cache::get('fb_metric:violations') ?: 0),
            ];

            $shim = [
                'routing_decision' => (int) (Cache::get('shim_metric:routing_decision') ?: 0),
                'rate_limit_exceeded' => (int) (Cache::get('shim_metric:rate_limit_exceeded') ?: 0),
                'fallback_shown' => (int) (Cache::get('shim_metric:fallback_shown') ?: 0),
                'signature_mismatch' => (int) (Cache::get('shim_metric:signature_mismatch') ?: 0),
            ];

            return $this->success(['facebook' => $facebook, 'shim' => $shim]);
        } catch (\Throwable $e) {
            return $this->error($e->getMessage());
        }
    }

    /**
     * Check domain health
     *
     * @return JsonResponse
     */
    public function checkDomainHealth(): JsonResponse
    {
        $domain = (string) ($_GET['domain'] ?? '');

        if ($domain === '') {
            return $this->error('Domain required');
        }

        try {
            $health = FacebookProtection::getDomainHealth($domain);
            return $this->success(['health' => $health]);
        } catch (\Throwable $e) {
            return $this->error($e->getMessage());
        }
    }

    /**
     * Get Shim configuration from .env
     *
     * @return JsonResponse
     */
    public function getShimConfig(): JsonResponse
    {
        try {
            $secret = Env::getString('SHIM_HMAC_SECRET', '');

            return $this->success([
                'config' => [
                    'SHIM_HMAC_SECRET_set' => $secret !== '',
                    'SHIM_ALLOWED_HOSTS' => Env::getString('SHIM_ALLOWED_HOSTS', ''),
                    'SHIM_ALLOWED_SUFFIXES' => Env::getString('SHIM_ALLOWED_SUFFIXES', ''),
                    'SHIM_VERSION' => Env::getString('SHIM_VERSION', 'v2'),
                ],
            ]);
        } catch (\Throwable $e) {
            return $this->error($e->getMessage());
        }
    }
}
