<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Env;
use App\Infrastructure\Http\JsonResponse;
use PDO;

/**
 * API controller for settings management
 */
final class SettingsApiController extends BaseApiController
{
    private string $dataDir;

    public function __construct()
    {
        parent::__construct();
        $this->dataDir = dirname(__DIR__, 4) . '/data';
    }

    /**
     * Get all settings from database
     *
     * @return JsonResponse
     */
    public function get(): JsonResponse
    {
        $stmt = $this->pdo->query('SELECT `key`, value FROM settings');
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $settings = [];
        foreach ($rows as $row) {
            $settings[$row['key']] = $row['value'];
        }

        return $this->success(['settings' => $settings]);
    }

    /**
     * Get public settings (redirect_url only)
     *
     * @return JsonResponse
     */
    public function getPublic(): JsonResponse
    {
        $settingsFile = $this->dataDir . '/settings.json';
        $settings = ['redirect_url' => ''];

        if (file_exists($settingsFile)) {
            $loaded = json_decode((string) file_get_contents($settingsFile), true);
            if (is_array($loaded) && isset($loaded['redirect_url'])) {
                $settings['redirect_url'] = $loaded['redirect_url'];
            }
        }

        return $this->success(['settings' => $settings]);
    }

    /**
     * Get security settings (admin only)
     *
     * @return JsonResponse
     */
    public function getSecurity(): JsonResponse
    {
        $securityFile = $this->dataDir . '/security.json';
        $defaultConfig = [
            'allowed_hosts' => [],
            'denied_hosts' => [],
            'denied_tlds' => [],
            'https_only' => true,
            'block_ip_literal' => true,
            'block_userinfo' => true,
            'ip_intel_enabled' => false,
            'risk_threshold' => 75,
            'safe_url' => '',
            'fallback_url' => '',
            'blackbox_api_url' => '',
            'ipquery_api_url' => '',
            'ipquery_api_key' => '',
            'atlas_api_url' => '',
            'sign_enabled' => false,
            'sign_secret' => '',
            'sign_ttl' => 3600,
        ];

        if (file_exists($securityFile)) {
            $loaded = json_decode((string) file_get_contents($securityFile), true);
            if (is_array($loaded)) {
                $defaultConfig = array_merge($defaultConfig, $loaded);
            }
        }

        // Override with .env if set
        if (Env::getString('SHORTLINK_ALLOWED_HOSTS', '') !== '') {
            $defaultConfig['allowed_hosts'] = Env::getList('SHORTLINK_ALLOWED_HOSTS');
        }
        if (Env::getString('SHORTLINK_DENIED_HOSTS', '') !== '') {
            $defaultConfig['denied_hosts'] = Env::getList('SHORTLINK_DENIED_HOSTS');
        }
        if (Env::getString('SHORTLINK_DENIED_TLDS', '') !== '') {
            $defaultConfig['denied_tlds'] = Env::getList('SHORTLINK_DENIED_TLDS');
        }

        return $this->success(['config' => $defaultConfig]);
    }

    /**
     * Update security settings (admin only)
     *
     * @return JsonResponse
     */
    public function updateSecurity(): JsonResponse
    {
        $data = $this->readJson();

        // Parse comma-separated lists
        $allowedHosts = $this->parseCommaSeparated($data['allowed_hosts'] ?? '');
        $deniedHosts = $this->parseCommaSeparated($data['denied_hosts'] ?? '');
        $deniedTlds = $this->parseCommaSeparated($data['denied_tlds'] ?? '');

        $config = [
            'allowed_hosts' => $allowedHosts,
            'denied_hosts' => $deniedHosts,
            'denied_tlds' => $deniedTlds,
            'https_only' => (bool) ($data['https_only'] ?? true),
            'block_ip_literal' => (bool) ($data['block_ip_literal'] ?? true),
            'block_userinfo' => (bool) ($data['block_userinfo'] ?? true),
            'ip_intel_enabled' => (bool) ($data['ip_intel_enabled'] ?? false),
            'risk_threshold' => max(0, min(100, (int) ($data['risk_threshold'] ?? 75))),
            'safe_url' => trim((string) ($data['safe_url'] ?? '')),
            'fallback_url' => trim((string) ($data['fallback_url'] ?? '')),
            'blackbox_api_url' => trim((string) ($data['blackbox_api_url'] ?? '')),
            'ipquery_api_url' => trim((string) ($data['ipquery_api_url'] ?? '')),
            'ipquery_api_key' => trim((string) ($data['ipquery_api_key'] ?? '')),
            'atlas_api_url' => trim((string) ($data['atlas_api_url'] ?? '')),
            'sign_enabled' => (bool) ($data['sign_enabled'] ?? false),
            'sign_secret' => trim((string) ($data['sign_secret'] ?? '')),
            'sign_ttl' => max(0, (int) ($data['sign_ttl'] ?? 3600)),
            'cpanel_host' => trim((string) ($data['cpanel_host'] ?? '')),
            'cpanel_username' => trim((string) ($data['cpanel_username'] ?? '')),
            'cpanel_token' => trim((string) ($data['cpanel_token'] ?? '')),
            'cf_api_token' => trim((string) ($data['cf_api_token'] ?? '')),
            'cf_default_ttl' => max(1, (int) ($data['cf_default_ttl'] ?? 1)),
            'cf_default_proxied' => (bool) ($data['cf_default_proxied'] ?? false),
            'rate_limit_enabled' => (bool) ($data['rate_limit_enabled'] ?? false),
            'rate_limit_max_requests' => max(1, (int) ($data['rate_limit_max_requests'] ?? 100)),
            'rate_limit_window' => max(1, (int) ($data['rate_limit_window'] ?? 60)),
            'rate_limit_by' => in_array($data['rate_limit_by'] ?? '', ['ip', 'user', 'ip_user'], true)
                ? $data['rate_limit_by'] : 'ip',
            'rate_limit_whitelist_admin' => (bool) ($data['rate_limit_whitelist_admin'] ?? true),
        ];

        $securityFile = $this->dataDir . '/security.json';

        if (!is_dir($this->dataDir)) {
            mkdir($this->dataDir, 0755, true);
        }

        $saved = file_put_contents(
            $securityFile,
            json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        );

        if ($saved === false) {
            return $this->error('Failed to save security settings');
        }

        return $this->success();
    }

    /**
     * Parse comma-separated string to array
     *
     * @param mixed $value
     * @return list<string>
     */
    private function parseCommaSeparated(mixed $value): array
    {
        if (!is_string($value)) {
            return [];
        }

        return array_values(array_filter(array_map('trim', explode(',', $value))));
    }
}
