<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Cache;
use App\Http;
use App\Infrastructure\Http\JsonResponse;
use App\ShortlinkRepository;

/**
 * API controller for shortlink CRUD operations
 */
final class ShortlinkApiController extends BaseApiController
{
    private ShortlinkRepository $repo;

    public function __construct()
    {
        parent::__construct();
        $this->repo = new ShortlinkRepository($this->pdo);
    }

    /**
     * List shortlinks with pagination and filters
     *
     * @return JsonResponse
     */
    public function list(): JsonResponse
    {
        $q = trim((string) ($_GET['q'] ?? ''));
        $status = (string) ($_GET['status'] ?? 'all');

        $page = max(1, (int) ($_GET['page'] ?? 1));
        $pageSize = min(100, max(1, (int) ($_GET['page_size'] ?? 20)));

        $isActive = null;
        if ($status === 'active') {
            $isActive = 1;
        } elseif ($status === 'inactive') {
            $isActive = 0;
        }

        $offset = ($page - 1) * $pageSize;
        $res = $this->repo->list($q, $isActive, $pageSize, $offset);

        return $this->success(['total' => $res['total'], 'rows' => $res['rows']]);
    }

    /**
     * Get single shortlink by code
     *
     * @return JsonResponse
     */
    public function get(): JsonResponse
    {
        $code = (string) ($_GET['code'] ?? '');
        $code = Http::validateCode($code);

        if ($code === '') {
            return $this->error('Invalid code');
        }

        $row = $this->repo->findByCode($code);
        if ($row === null) {
            return $this->error('Not found', 404);
        }

        return $this->success(['row' => $row]);
    }

    /**
     * Create new shortlink
     *
     * @return JsonResponse
     */
    public function create(): JsonResponse
    {
        $data = $this->readJson();

        $codeRaw = is_string($data['code'] ?? null) ? trim($data['code']) : '';
        $code = $codeRaw !== '' ? Http::validateCode($codeRaw) : $this->generateCode();

        if ($code === '') {
            return $this->error('Invalid code');
        }

        if ($this->repo->exists($code)) {
            return $this->error('Code already exists');
        }

        $insert = $this->validatePayload($data);
        if ($insert === null) {
            return $this->error('Invalid payload');
        }

        $this->repo->insert(array_merge(['code' => $code], $insert));

        return $this->success(['code' => $code]);
    }

    /**
     * Update existing shortlink
     *
     * @return JsonResponse
     */
    public function update(): JsonResponse
    {
        $data = $this->readJson();

        $code = is_string($data['code'] ?? null) ? $data['code'] : '';
        $code = Http::validateCode($code);

        if ($code === '') {
            return $this->error('Invalid code');
        }

        $payload = $this->validatePayload($data);
        if ($payload === null) {
            return $this->error('Invalid payload');
        }

        if ($this->repo->findByCode($code) === null) {
            return $this->error('Not found', 404);
        }

        $this->repo->updateByCode($code, $payload);
        Cache::invalidateShortlink($code);

        return $this->success();
    }

    /**
     * Set shortlink active/inactive status
     *
     * @return JsonResponse
     */
    public function setActive(): JsonResponse
    {
        $data = $this->readJson();

        $code = is_string($data['code'] ?? null) ? $data['code'] : '';
        $code = Http::validateCode($code);

        if ($code === '') {
            return $this->error('Invalid code');
        }

        $isActive = is_int($data['is_active'] ?? null) ? $data['is_active'] : 0;
        $isActive = $isActive === 1 ? 1 : 0;

        if ($this->repo->findByCode($code) === null) {
            return $this->error('Not found', 404);
        }

        $this->repo->setActive($code, $isActive);
        Cache::invalidateShortlink($code);

        return $this->success();
    }

    /**
     * Delete single shortlink
     *
     * @return JsonResponse
     */
    public function delete(): JsonResponse
    {
        $data = $this->readJson();

        $code = is_string($data['code'] ?? null) ? $data['code'] : '';
        $code = Http::validateCode($code);

        if ($code === '') {
            return $this->error('Invalid code');
        }

        if ($this->repo->findByCode($code) === null) {
            return $this->error('Shortlink not found', 404);
        }

        $stmt = $this->pdo->prepare('DELETE FROM shortlinks WHERE code = ?');
        $stmt->execute([$code]);

        Cache::invalidateShortlink($code);

        return $this->success();
    }

    /**
     * Bulk delete shortlinks
     *
     * @return JsonResponse
     */
    public function bulkDelete(): JsonResponse
    {
        $data = $this->readJson();

        $codes = $data['codes'] ?? [];
        if (!is_array($codes) || empty($codes)) {
            return $this->error('No codes provided');
        }

        $validCodes = [];
        foreach ($codes as $code) {
            if (is_string($code)) {
                $validated = Http::validateCode($code);
                if ($validated !== '') {
                    $validCodes[] = $validated;
                }
            }
        }

        if (empty($validCodes)) {
            return $this->error('No valid codes provided');
        }

        if (count($validCodes) > 100) {
            return $this->error('Too many codes (max 100)');
        }

        $placeholders = str_repeat('?,', count($validCodes) - 1) . '?';
        $stmt = $this->pdo->prepare("DELETE FROM shortlinks WHERE code IN ($placeholders)");
        $stmt->execute($validCodes);
        $deletedCount = $stmt->rowCount();

        foreach ($validCodes as $code) {
            Cache::invalidateShortlink($code);
        }

        return $this->success([
            'deleted' => $deletedCount,
            'requested' => count($validCodes),
        ]);
    }

    /**
     * Generate a unique random code
     */
    private function generateCode(): string
    {
        for ($i = 0; $i < 5; $i++) {
            $raw = rtrim(strtr(base64_encode(random_bytes(6)), '+/', '-_'), '=');
            $code = substr($raw, 0, 10);
            $code = Http::validateCode($code);

            if ($code !== '' && !$this->repo->exists($code)) {
                return $code;
            }
        }

        return '';
    }

    /**
     * Validate and sanitize payload
     *
     * @param array<string, mixed> $data
     * @return array{og_title:string,og_description:string,og_image_url:string,og_type:string,og_site_name:string,og_locale:string,use_shim:int,domain_type:?string,domain_id:?int}|null
     */
    private function validatePayload(array $data): ?array
    {
        $title = trim(is_string($data['og_title'] ?? null) ? $data['og_title'] : '');
        $desc = trim(is_string($data['og_description'] ?? null) ? $data['og_description'] : '');
        $type = trim(is_string($data['og_type'] ?? null) ? $data['og_type'] : 'website');
        $site = trim(is_string($data['og_site_name'] ?? null) ? $data['og_site_name'] : '');
        $loc = trim(is_string($data['og_locale'] ?? null) ? $data['og_locale'] : 'id_ID');

        if ($type === '') {
            $type = 'website';
        }
        if ($loc === '') {
            $loc = 'id_ID';
        }

        $imgRaw = trim(is_string($data['og_image_url'] ?? null) ? $data['og_image_url'] : '');
        $img = $imgRaw !== '' ? Http::validateUrl($imgRaw, false) : '';

        $useShim = isset($data['use_shim']) ? ((int) $data['use_shim'] === 1 ? 1 : 0) : 0;

        $domainType = null;
        $domainId = null;

        if (isset($data['domain_type']) && is_string($data['domain_type']) && $data['domain_type'] !== '') {
            $domainType = in_array($data['domain_type'], ['global', 'user'], true) ? $data['domain_type'] : null;
        }

        if (isset($data['domain_id']) && $data['domain_id'] !== null) {
            $domainId = (int) $data['domain_id'];
            if ($domainId <= 0) {
                $domainId = null;
            }
        }

        return [
            'og_title' => mb_substr($title, 0, 255),
            'og_description' => mb_substr($desc, 0, 600),
            'og_image_url' => $img,
            'og_type' => mb_substr($type, 0, 32),
            'og_site_name' => mb_substr($site, 0, 64),
            'og_locale' => mb_substr($loc, 0, 16),
            'use_shim' => $useShim,
            'domain_type' => $domainType,
            'domain_id' => $domainId,
        ];
    }
}
