<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Infrastructure\Http\JsonResponse;

/**
 * API controller for third-party URL shortening services
 */
final class ThirdPartyApiController extends BaseApiController
{
    /**
     * Create shortlink using third-party service
     *
     * @return JsonResponse
     */
    public function createShortlink(): JsonResponse
    {
        $data = $this->readJson();

        $service = trim((string) ($data['service'] ?? ''));
        $longUrl = trim((string) ($data['long_url'] ?? ''));

        if ($service === '' || $longUrl === '') {
            return $this->error('Service and long_url are required');
        }

        if (!filter_var($longUrl, FILTER_VALIDATE_URL)) {
            return $this->error('Invalid URL');
        }

        $ch = null;
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);

            return match ($service) {
                'isgd' => $this->callIsGd($ch, $longUrl),
                'vgd' => $this->callVGd($ch, $longUrl),
                'tinyurl' => $this->callTinyUrl($ch, $longUrl),
                'dagd' => $this->callDaGd($ch, $longUrl),
                'clckru' => $this->callClckRu($ch, $longUrl),
                default => $this->error('Unknown service: ' . $service),
            };
        } catch (\Throwable $e) {
            error_log('3rd party API error: ' . $e->getMessage());
            return $this->error('Server error: ' . $e->getMessage());
        } finally {
            if ($ch !== null) {
                curl_close($ch);
            }
        }
    }

    /**
     * Call is.gd API
     *
     * @param \CurlHandle $ch
     * @param string $longUrl
     * @return JsonResponse
     */
    private function callIsGd(\CurlHandle $ch, string $longUrl): JsonResponse
    {
        $apiUrl = 'https://is.gd/create.php?format=json&url=' . urlencode($longUrl);
        curl_setopt($ch, CURLOPT_URL, $apiUrl);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($httpCode !== 200 || !$response) {
            return $this->error('is.gd API request failed');
        }

        $data = json_decode((string) $response, true);
        if (isset($data['shorturl'])) {
            return $this->success(['url' => $data['shorturl']]);
        }

        return $this->error($data['errormessage'] ?? 'is.gd API error');
    }

    /**
     * Call v.gd API
     *
     * @param \CurlHandle $ch
     * @param string $longUrl
     * @return JsonResponse
     */
    private function callVGd(\CurlHandle $ch, string $longUrl): JsonResponse
    {
        $apiUrl = 'https://v.gd/create.php?format=json&url=' . urlencode($longUrl);
        curl_setopt($ch, CURLOPT_URL, $apiUrl);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($httpCode !== 200 || !$response) {
            return $this->error('v.gd API request failed');
        }

        $data = json_decode((string) $response, true);
        if (isset($data['shorturl'])) {
            return $this->success(['url' => $data['shorturl']]);
        }

        return $this->error($data['errormessage'] ?? 'v.gd API error');
    }

    /**
     * Call TinyURL API
     *
     * @param \CurlHandle $ch
     * @param string $longUrl
     * @return JsonResponse
     */
    private function callTinyUrl(\CurlHandle $ch, string $longUrl): JsonResponse
    {
        $apiUrl = 'https://tinyurl.com/api-create.php?url=' . urlencode($longUrl);
        curl_setopt($ch, CURLOPT_URL, $apiUrl);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($httpCode !== 200 || !$response) {
            return $this->error('TinyURL API request failed');
        }

        $shortUrl = trim((string) $response);
        if ($shortUrl && str_starts_with($shortUrl, 'http')) {
            return $this->success(['url' => $shortUrl]);
        }

        return $this->error('TinyURL API error');
    }

    /**
     * Call da.gd API
     *
     * @param \CurlHandle $ch
     * @param string $longUrl
     * @return JsonResponse
     */
    private function callDaGd(\CurlHandle $ch, string $longUrl): JsonResponse
    {
        $apiUrl = 'https://da.gd/s?url=' . urlencode($longUrl);
        curl_setopt($ch, CURLOPT_URL, $apiUrl);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($httpCode !== 200 || !$response) {
            return $this->error('da.gd API request failed');
        }

        $shortUrl = trim((string) $response);
        if ($shortUrl && str_starts_with($shortUrl, 'http')) {
            return $this->success(['url' => $shortUrl]);
        }

        return $this->error('da.gd API error');
    }

    /**
     * Call clck.ru API
     *
     * @param \CurlHandle $ch
     * @param string $longUrl
     * @return JsonResponse
     */
    private function callClckRu(\CurlHandle $ch, string $longUrl): JsonResponse
    {
        $apiUrl = 'https://clck.ru/--?url=' . urlencode($longUrl);
        curl_setopt($ch, CURLOPT_URL, $apiUrl);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($httpCode !== 200 || !$response) {
            return $this->error('clck.ru API request failed');
        }

        $shortUrl = trim((string) $response);
        if ($shortUrl && str_starts_with($shortUrl, 'http')) {
            return $this->success(['url' => $shortUrl]);
        }

        return $this->error('clck.ru API error');
    }
}
