<?php

declare(strict_types=1);

namespace App\Presentation\Controller\Api;

use App\Infrastructure\Http\JsonResponse;
use PDO;

/**
 * API controller for user management (admin only)
 */
final class UserApiController extends BaseApiController
{
    /**
     * List all users
     *
     * @return JsonResponse
     */
    public function list(): JsonResponse
    {
        $stmt = $this->pdo->query(
            'SELECT id, username, name, role, is_active, created_at, updated_at FROM users ORDER BY id ASC'
        );
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return $this->success(['users' => $users]);
    }

    /**
     * Get single user by ID
     *
     * @return JsonResponse
     */
    public function get(): JsonResponse
    {
        $id = (int) ($_GET['id'] ?? 0);
        if ($id < 1) {
            return $this->error('Invalid user ID');
        }

        $stmt = $this->pdo->prepare(
            'SELECT id, username, name, role, is_active, created_at, updated_at FROM users WHERE id = ?'
        );
        $stmt->execute([$id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!is_array($user)) {
            return $this->error('User not found', 404);
        }

        return $this->success(['user' => $user]);
    }

    /**
     * Create new user
     *
     * @return JsonResponse
     */
    public function create(): JsonResponse
    {
        $data = $this->readJson();

        $username = trim((string) ($data['username'] ?? ''));
        $password = trim((string) ($data['password'] ?? ''));
        $name = trim((string) ($data['name'] ?? ''));
        $role = trim((string) ($data['role'] ?? 'user'));
        $isActive = isset($data['is_active']) ? (int) $data['is_active'] : 1;

        if ($username === '' || $password === '' || $name === '') {
            return $this->error('Username, password, and name are required');
        }

        if (!in_array($role, ['admin', 'user'], true)) {
            return $this->error('Invalid role');
        }

        // Check if username exists
        $stmt = $this->pdo->prepare('SELECT id FROM users WHERE username = ?');
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            return $this->error('Username already exists');
        }

        // Hash password
        $hash = password_hash($password, PASSWORD_BCRYPT);

        // Insert user
        $stmt = $this->pdo->prepare(
            'INSERT INTO users (username, password_hash, name, role, is_active) VALUES (?, ?, ?, ?, ?)'
        );
        $stmt->execute([$username, $hash, $name, $role, $isActive]);

        $userId = (int) $this->pdo->lastInsertId();

        return $this->success(['user_id' => $userId]);
    }

    /**
     * Update existing user
     *
     * @return JsonResponse
     */
    public function update(): JsonResponse
    {
        $data = $this->readJson();

        $id = (int) ($data['id'] ?? 0);
        if ($id < 1) {
            return $this->error('Invalid user ID');
        }

        // Check if user exists
        $stmt = $this->pdo->prepare('SELECT id FROM users WHERE id = ?');
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            return $this->error('User not found', 404);
        }

        $updates = [];
        $params = [];

        if (isset($data['username'])) {
            $username = trim((string) $data['username']);
            if ($username === '') {
                return $this->error('Username cannot be empty');
            }

            // Check if username exists for other user
            $stmt = $this->pdo->prepare('SELECT id FROM users WHERE username = ? AND id != ?');
            $stmt->execute([$username, $id]);
            if ($stmt->fetch()) {
                return $this->error('Username already exists');
            }

            $updates[] = 'username = ?';
            $params[] = $username;
        }

        if (isset($data['name'])) {
            $name = trim((string) $data['name']);
            if ($name === '') {
                return $this->error('Name cannot be empty');
            }
            $updates[] = 'name = ?';
            $params[] = $name;
        }

        if (isset($data['password']) && $data['password'] !== '') {
            $password = trim((string) $data['password']);
            $hash = password_hash($password, PASSWORD_BCRYPT);
            $updates[] = 'password_hash = ?';
            $params[] = $hash;
        }

        if (isset($data['role'])) {
            $role = trim((string) $data['role']);
            if (!in_array($role, ['admin', 'user'], true)) {
                return $this->error('Invalid role');
            }
            $updates[] = 'role = ?';
            $params[] = $role;
        }

        if (isset($data['is_active'])) {
            $isActive = $data['is_active'] ? 1 : 0;
            $updates[] = 'is_active = ?';
            $params[] = $isActive;
        }

        if (empty($updates)) {
            return $this->error('No fields to update');
        }

        $params[] = $id;
        $sql = 'UPDATE users SET ' . implode(', ', $updates) . ' WHERE id = ?';
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        return $this->success();
    }

    /**
     * Delete user
     *
     * @return JsonResponse
     */
    public function delete(): JsonResponse
    {
        $data = $this->readJson();

        $id = (int) ($data['id'] ?? 0);
        if ($id < 1) {
            return $this->error('Invalid user ID');
        }

        // Prevent deleting own account
        if ($this->getUserId() === $id) {
            return $this->error('Cannot delete your own account');
        }

        // Check if user exists
        $stmt = $this->pdo->prepare('SELECT id FROM users WHERE id = ?');
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            return $this->error('User not found', 404);
        }

        // Delete user
        $stmt = $this->pdo->prepare('DELETE FROM users WHERE id = ?');
        $stmt->execute([$id]);

        return $this->success();
    }

    /**
     * Bulk delete users
     *
     * @return JsonResponse
     */
    public function bulkDelete(): JsonResponse
    {
        $data = $this->readJson();

        $ids = $data['ids'] ?? [];
        if (!is_array($ids) || empty($ids)) {
            return $this->error('No user IDs provided');
        }

        // Validate and filter IDs
        $validIds = [];
        foreach ($ids as $id) {
            if (is_int($id) || (is_string($id) && ctype_digit($id))) {
                $validIds[] = (int) $id;
            }
        }

        if (empty($validIds)) {
            return $this->error('No valid user IDs provided');
        }

        if (count($validIds) > 50) {
            return $this->error('Too many users (max 50)');
        }

        // Prevent deleting own account
        $currentUserId = $this->getUserId();
        if (in_array($currentUserId, $validIds, true)) {
            return $this->error('Cannot delete your own account');
        }

        $placeholders = str_repeat('?,', count($validIds) - 1) . '?';
        $stmt = $this->pdo->prepare("DELETE FROM users WHERE id IN ($placeholders)");
        $stmt->execute($validIds);
        $deletedCount = $stmt->rowCount();

        return $this->success([
            'deleted' => $deletedCount,
            'requested' => count($validIds),
        ]);
    }

    /**
     * Toggle user active status
     *
     * @return JsonResponse
     */
    public function toggleActive(): JsonResponse
    {
        $data = $this->readJson();

        $id = (int) ($data['id'] ?? 0);
        if ($id < 1) {
            return $this->error('Invalid user ID');
        }

        $isActive = (int) ($data['is_active'] ?? 1);
        if (!in_array($isActive, [0, 1], true)) {
            return $this->error('Invalid status value');
        }

        // Prevent deactivating own account
        if ($this->getUserId() === $id && $isActive === 0) {
            return $this->error('Cannot deactivate your own account');
        }

        // Check if user exists
        $stmt = $this->pdo->prepare('SELECT id FROM users WHERE id = ?');
        $stmt->execute([$id]);
        if (!$stmt->fetch()) {
            return $this->error('User not found', 404);
        }

        // Update user status
        $stmt = $this->pdo->prepare('UPDATE users SET is_active = ? WHERE id = ?');
        $stmt->execute([$isActive, $id]);

        return $this->success();
    }
}
