<?php

declare(strict_types=1);

namespace App\Presentation\Controller;

use App\Env;
use App\Http;

/**
 * Renders OpenGraph HTML page for Facebook crawler
 */
final class OgPageRenderer
{
    /**
     * Render OG page HTML
     *
     * @param array{
     *   code: string,
     *   og_title: string,
     *   og_description: string,
     *   og_image_url: string,
     *   og_type: string,
     *   og_site_name: string,
     *   og_locale: string
     * } $shortlink Shortlink data
     * @param string $finalUrl Target URL for redirect
     * @param string $nonce CSP nonce
     * @return string HTML content
     */
    public function render(array $shortlink, string $finalUrl, string $nonce): string
    {
        $publicBase = Env::getString('PUBLIC_BASE_URL', '');
        $shortUrl = $publicBase !== ''
            ? (rtrim($publicBase, '/') . '/' . $shortlink['code'])
            : $shortlink['code'];

        $fbAppId = Env::getString('FB_APP_ID', '');

        $ogTitle = $shortlink['og_title'];
        $ogDesc = $shortlink['og_description'];
        $ogType = $shortlink['og_type'] !== '' ? $shortlink['og_type'] : 'website';
        $ogSite = $shortlink['og_site_name'];
        $ogLoc = $shortlink['og_locale'] !== '' ? $shortlink['og_locale'] : 'id_ID';
        $ogImg = Http::validateUrl($shortlink['og_image_url'], false);

        $html = '<!doctype html>';
        $html .= '<html lang="id"><head>';
        $html .= '<meta charset="utf-8">';
        $html .= '<meta name="viewport" content="width=device-width, initial-scale=1">';

        // OpenGraph meta tags
        $html .= '<meta property="og:title" content="' . Http::e($ogTitle) . '">';
        $html .= '<meta property="og:description" content="' . Http::e($ogDesc) . '">';
        $html .= '<meta property="og:type" content="' . Http::e($ogType) . '">';
        $html .= '<meta property="og:locale" content="' . Http::e($ogLoc) . '">';
        $html .= '<meta property="og:url" content="' . Http::e($shortUrl) . '">';

        if ($ogSite !== '') {
            $html .= '<meta property="og:site_name" content="' . Http::e($ogSite) . '">';
        }

        if ($ogImg !== '') {
            $html .= '<meta property="og:image" content="' . Http::e($ogImg) . '">';
            $html .= '<meta property="og:image:secure_url" content="' . Http::e($ogImg) . '">';
        }

        if ($fbAppId !== '') {
            $html .= '<meta property="fb:app_id" content="' . Http::e($fbAppId) . '">';
        }

        $html .= '<link rel="canonical" href="' . Http::e($shortUrl) . '">';
        $html .= '<title>' . Http::e($ogTitle) . '</title>';

        // Inline styles with nonce
        $html .= '<style nonce="' . Http::e($nonce) . '">';
        $html .= 'body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;margin:24px;color:#111}';
        $html .= '.box{max-width:720px;padding:16px;border:1px solid #ddd;border-radius:10px}';
        $html .= 'a{color:#111}';
        $html .= '</style>';

        $html .= '</head><body>';
        $html .= '<div class="box">';
        $html .= '<h1 style="margin:0 0 8px 0;font-size:18px;">' . Http::e($ogTitle) . '</h1>';
        $html .= '<p style="margin:0 0 12px 0;opacity:.8;">' . Http::e($ogDesc) . '</p>';
        $html .= '<p style="margin:0;">Target: <a href="' . Http::e($finalUrl) . '">' . Http::e($finalUrl) . '</a></p>';
        $html .= '</div>';
        $html .= '</body></html>';

        return $html;
    }
}
