<?php

declare(strict_types=1);

namespace App\Presentation\Controller;

use App\Analytics\AsyncTrafficLogger;
use App\Application\Shortlink\ShortlinkResolver;
use App\Env;
use App\FacebookProtection;
use App\Http;
use App\Infrastructure\Http\JsonResponse;
use App\Infrastructure\Http\RedirectResponse;
use App\Infrastructure\Http\Request;
use App\Infrastructure\Http\Response;
use App\IpResolver;
use App\RedirectLogger;
use App\RedirectRouter;
use App\ShimLink;
use App\SignedUrl;
use App\UrlValidator;
use RuntimeException;
use Throwable;

/**
 * Controller for handling shortlink redirects
 */
final class RedirectController
{
    private ShortlinkResolver $resolver;
    private OgPageRenderer $ogRenderer;

    public function __construct(
        ?ShortlinkResolver $resolver = null,
        ?OgPageRenderer $ogRenderer = null
    ) {
        $this->resolver = $resolver ?? new ShortlinkResolver();
        $this->ogRenderer = $ogRenderer ?? new OgPageRenderer();
    }

    /**
     * Handle the redirect request
     */
    public function handle(Request $request): Response
    {
        $nonce = Http::nonce();

        try {
            // 1. Resolve client IP
            $clientIp = IpResolver::resolve();

            // 2. Get and validate shortlink code
            $code = $this->resolveCode($request);
            if ($code === null) {
                return new Response('Not found', 404);
            }

            // 3. Resolve shortlink data
            $shortlink = $this->resolver->resolve($code, $clientIp);
            if ($shortlink === null) {
                return new Response('Not found', 404);
            }

            // 4. Get redirect URL from settings
            $redirectUrl = $this->getRedirectUrl();
            if ($redirectUrl === '') {
                return new Response('Global redirect URL not configured', 500);
            }

            // 5. Validate target URL
            $targetUrl = $this->validateTargetUrl($redirectUrl, $code, $clientIp);
            if ($targetUrl === null) {
                return new Response('Target URL not allowed', 403);
            }

            // 6. Apply IP intelligence routing
            $routeResult = $this->applyIpIntelRouting($targetUrl, $clientIp, $code);
            $finalUrl = $routeResult['url'];

            // 7. Apply Facebook protection
            $fbResult = $this->applyFacebookProtection(
                $request,
                $code,
                $clientIp,
                $finalUrl
            );

            if ($fbResult['blocked']) {
                return new Response($fbResult['message'], $fbResult['status_code'], $fbResult['headers']);
            }

            // 8. Check if Facebook bot - render OG page
            if ($fbResult['is_fb_bot']) {
                Http::sendSecurityHeaders($nonce, true);
                $html = $this->ogRenderer->render($shortlink, $finalUrl, $nonce);
                return new Response($html, 200, ['Content-Type' => 'text/html; charset=utf-8']);
            }

            // 9. Normal redirect
            Http::sendSecurityHeaders($nonce, false);

            // 10. Log traffic asynchronously (non-blocking)
            $this->logTrafficAsync(
                $code,
                $clientIp,
                $finalUrl,
                $request->getUserAgent(),
                $request->getHeader('Referer')
            );

            // Apply shim if enabled
            $useShim = ($shortlink['use_shim'] === 1) || ShimLink::isEnabled();
            if ($useShim) {
                try {
                    $shimUrl = ShimLink::generate($finalUrl);
                    return RedirectResponse::temporary($shimUrl);
                } catch (RuntimeException $e) {
                    error_log('shim_generation_failed: ' . $e->getMessage());
                }
            }

            return RedirectResponse::temporary($finalUrl);

        } catch (Throwable $e) {
            return $this->handleError($e);
        }
    }

    /**
     * Resolve shortlink code from request
     */
    private function resolveCode(Request $request): ?string
    {
        $signedUrlEnabled = Env::getBool('URL_SIGN_ENABLED', false);

        if ($signedUrlEnabled) {
            $verification = SignedUrl::verify($request->getAllQuery());
            if (!$verification['valid']) {
                return null;
            }
            return $verification['code'];
        }

        $code = $request->getQuery('code');
        $code = Http::validateCode($code);

        return $code !== '' ? $code : null;
    }

    /**
     * Get redirect URL from settings
     */
    private function getRedirectUrl(): string
    {
        $settingsFile = dirname(__DIR__, 3) . '/data/settings.json';

        if (!file_exists($settingsFile)) {
            return '';
        }

        $content = file_get_contents($settingsFile);
        if ($content === false) {
            return '';
        }

        $settings = json_decode($content, true);
        if (!is_array($settings) || !isset($settings['redirect_url'])) {
            return '';
        }

        return (string) $settings['redirect_url'];
    }

    /**
     * Validate target URL
     */
    private function validateTargetUrl(string $redirectUrl, string $code, string $clientIp): ?string
    {
        $allowedHosts = Env::getList('SHORTLINK_ALLOWED_HOSTS');
        $deniedHosts = Env::getList('SHORTLINK_DENIED_HOSTS');
        $deniedTlds = Env::getList('SHORTLINK_DENIED_TLDS');

        if (empty($deniedTlds)) {
            $deniedTlds = UrlValidator::getDefaultDeniedTlds();
        }

        $httpsOnly = Env::getBool('SHORTLINK_HTTPS_ONLY', true);
        $blockIpLiteral = Env::getBool('SHORTLINK_BLOCK_IP_LITERAL', true);
        $blockUserinfo = Env::getBool('SHORTLINK_BLOCK_USERINFO', true);

        $validation = UrlValidator::validate(
            $redirectUrl,
            $allowedHosts,
            $deniedHosts,
            $deniedTlds,
            $httpsOnly,
            $blockIpLiteral,
            $blockUserinfo
        );

        if (!$validation['valid']) {
            $logger = new RedirectLogger();
            $logger->logBlocked($code, $clientIp, 'url_validation_failed', [
                'error' => $validation['error'],
                'target_url' => $redirectUrl,
            ]);
            return null;
        }

        return $validation['url'];
    }

    /**
     * Apply IP intelligence routing
     *
     * @return array{url: string, route_type: string, risk_data: array<string, mixed>}
     */
    private function applyIpIntelRouting(string $targetUrl, string $clientIp, string $code): array
    {
        $ipIntelEnabled = Env::getBool('IP_INTEL_ENABLED', false);

        if (!$ipIntelEnabled) {
            return [
                'url' => $targetUrl,
                'route_type' => 'direct',
                'risk_data' => ['is_risky' => false, 'risk_score' => 0],
            ];
        }

        $safeUrl = Env::getString('SHORTLINK_SAFE_URL', $targetUrl);
        $fallbackUrl = Env::getString('SHORTLINK_FALLBACK_URL', $targetUrl);

        $router = new RedirectRouter($targetUrl, $safeUrl, $fallbackUrl, true);
        return $router->route($clientIp, $code);
    }

    /**
     * Apply Facebook protection
     *
     * @return array{
     *   blocked: bool,
     *   is_fb_bot: bool,
     *   message: string,
     *   status_code: int,
     *   headers: array<string, string>
     * }
     */
    private function applyFacebookProtection(
        Request $request,
        string $code,
        string $clientIp,
        string $finalUrl
    ): array {
        $ua = $request->getUserAgent();
        $isFbBot = Http::isFacebookBot($ua);

        $result = [
            'blocked' => false,
            'is_fb_bot' => $isFbBot,
            'message' => '',
            'status_code' => 200,
            'headers' => [],
        ];

        // Load Facebook protection config
        $configFile = dirname(__DIR__, 3) . '/config/facebook_protection.php';
        $fbProtectionConfig = file_exists($configFile) ? require $configFile : ['enabled' => true];
        $fbProtectionEnabled = $fbProtectionConfig['enabled'] ?? true;

        // Get current domain
        $publicBase = Env::getString('PUBLIC_BASE_URL', '');
        $currentDomain = '';
        if ($publicBase !== '') {
            $baseParts = parse_url($publicBase);
            if (is_array($baseParts) && isset($baseParts['host'])) {
                $currentDomain = (string) $baseParts['host'];
            }
        }

        if (!$fbProtectionEnabled || $currentDomain === '') {
            return $result;
        }

        // Validate destination URL
        $destValidation = FacebookProtection::validateDestination($finalUrl);
        if (!$destValidation['safe']) {
            $logger = new RedirectLogger();
            $logger->logBlocked($code, $clientIp, 'fb_protection_destination_unsafe', [
                'reason' => $destValidation['reason'],
                'target_url' => $finalUrl,
            ]);

            $result['blocked'] = true;
            $result['message'] = 'Destination URL flagged as unsafe';
            $result['status_code'] = 403;
            return $result;
        }

        // For Facebook bot traffic
        if ($isFbBot) {
            // Rate limiting
            if (!FacebookProtection::checkRateLimit($currentDomain, $ua)) {
                FacebookProtection::cooldownDomain($currentDomain, 600);

                $logger = new RedirectLogger();
                $logger->logBlocked($code, $clientIp, 'fb_rate_limit_exceeded', [
                    'domain' => $currentDomain,
                    'user_agent' => $ua,
                ]);

                $result['blocked'] = true;
                $result['message'] = 'Too many requests. Please try again later.';
                $result['status_code'] = 429;
                $result['headers']['Retry-After'] = '600';
                return $result;
            }

            // Domain health check
            $domainHealth = FacebookProtection::isDomainSafe($currentDomain);
            if (!$domainHealth['safe']) {
                $logger = new RedirectLogger();
                $logger->logBlocked($code, $clientIp, 'fb_domain_unhealthy', [
                    'domain' => $currentDomain,
                    'reason' => $domainHealth['reason'],
                ]);
                error_log('fb_domain_warning: ' . $currentDomain . ' - ' . $domainHealth['reason']);
            }

            // Validate Facebook IP
            if (!FacebookProtection::isFacebookIP($clientIp)) {
                error_log('fb_suspicious_request: FB bot UA from non-FB IP ' . $clientIp . ' UA=' . $ua);
            }
        }

        // Set response headers
        $responseHeaders = FacebookProtection::getResponseHeaders($ua);
        $result['headers']['Cache-Control'] = $responseHeaders['cache_control'];
        $result['headers']['Vary'] = $responseHeaders['vary_header'];

        return $result;
    }

    /**
     * Log traffic asynchronously
     */
    private function logTrafficAsync(
        string $code,
        string $clientIp,
        string $targetUrl,
        ?string $userAgent,
        ?string $referer
    ): void {
        // Skip if traffic logging is disabled
        if (!Env::getBool('TRAFFIC_LOG_ENABLED', true)) {
            return;
        }

        try {
            AsyncTrafficLogger::log([
                'shortlink_code' => $code,
                'target_url' => $targetUrl,
                'redirect_url' => $targetUrl,
                'ip_address' => $clientIp,
                'user_agent' => $userAgent,
                'referer' => $referer,
                'http_method' => $_SERVER['REQUEST_METHOD'] ?? 'GET',
                'http_protocol' => $_SERVER['SERVER_PROTOCOL'] ?? null,
            ]);
        } catch (Throwable $e) {
            // Don't let logging failures affect the redirect
            error_log('traffic_log_error: ' . $e->getMessage());
        }
    }

    /**
     * Handle error response
     */
    private function handleError(Throwable $e): Response
    {
        $rid = bin2hex(random_bytes(8));
        error_log('shortlink_error rid=' . $rid . ' type=' . get_class($e) . ' msg=' . $e->getMessage());

        return new Response('Server error', 500, ['X-Request-ID' => $rid]);
    }
}
